<?php

/**
 * Field Select Image
 *
 * @package     WordPress
 * @subpackage  ReduxFramework
 * @since       3.1.2
 * @author      Kevin Provance <kprovance>
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'ReduxFramework_select_image_new' ) ) {
	class ReduxFramework_select_image_new {

		public $parent;
		public $field;
		public $value;
		public $extension_dir;
		public $extension_url;

		/**
		 * Field Constructor.
		 * Required - must call the parent constructor, then assign field and value to vars, and obviously call the render field function
		 *
		 * @since ReduxFramework 1.0.0
		 */
		function __construct( $field, $value, $parent ) {
			$this->parent = $parent;
			$this->field  = $field;
			$this->value  = $value;

			if ( empty( $this->extension_dir ) ) {
				$this->extension_dir = trailingslashit( str_replace( '\\', '/', dirname( __FILE__ ) ) );
				$this->extension_url = plugin_dir_url( __FILE__ );
			}
		}

		/**
		 * Field Render Function.
		 * Takes the vars and outputs the HTML for the field in the settings
		 *
		 * @since ReduxFramework 1.0.0
		 */
		function render() {

			// If options is NOT empty, the process
			if ( ! empty( $this->field['options'] ) ) {

				// Strip off the file ext
				if ( isset( $this->value ) ) {
					$name = explode( '.', $this->value );
					$name = str_replace( '.' . end( $name ), '', $this->value );
					$name = basename( $name );
					//$this->value = trim( $name );
					$filename = trim( $name );
				}

				// beancounter
				$x = 1;

				// Process width
				if ( ! empty( $this->field['width'] ) ) {
					$width = ' style="width:' . $this->field['width'] . ';"';
				} else {
					$width = ' style="width: 40%;"';
				}

				// Process placeholder
				$placeholder = ( isset( $this->field['placeholder'] ) ) ? esc_attr( $this->field['placeholder'] ) : esc_html__( 'Select an item', 'pgs-core' );

				if ( isset( $this->field['select2'] ) ) { // if there are any let's pass them to js
					$select2_params = json_encode( $this->field['select2'] );
					$select2_params = htmlspecialchars( $select2_params, ENT_QUOTES );

					echo '<input type="hidden" class="select2_params" value="' . $select2_params . '">';
				}

				// Begin the <select> tag
				echo '<select data-id="' . $this->field['id'] . '" data-placeholder="' . $placeholder . '" name="' . $this->field['name'] . $this->field['name_suffix'] . '" class="redux-select-item redux-select-images ' . $this->field['class'] . '"' . $width . ' rows="6">';
				echo '<option></option>';

				$option_sr = 1;
				// Enum through the options array
				foreach ( $this->field['options'] as $k => $v ) {

					// No array?  No problem!
					if ( ! is_array( $v ) ) {
						$v = array( 'img' => $v );
					}

					// No title set?  Make it blank.
					if ( ! isset( $v['title'] ) && ! isset( $v['alt'] ) ) {
						$img_pathinfo = pathinfo( $v['img'] );
						$v['title']   = ucwords( str_replace( '-', ' ', $img_pathinfo['filename'] ) );
						$v['alt']     = sanitize_title( $img_pathinfo['filename'] );
					} elseif ( isset( $v['title'] ) && ! isset( $v['alt'] ) ) {
						$v['alt'] = sanitize_title( $v['title'] );
					} elseif ( ! isset( $v['title'] ) && isset( $v['alt'] ) ) {
						$v['title'] = ucwords( str_replace( '-', ' ', $v['alt'] ) );
					}

					// Set the selected entry
					$selected = selected( $this->value, $k, false );

					// If selected returns something other than a blank space, we
					// found our default/saved name.  Save the array number in a
					// variable to use later on when we want to extract its associted
					// url.
					if ( '' != $selected ) {
						$arrNum = $x;
					}

					// Add the option tag, with values.
					echo '<option value="' . $k . '" data-src="' . $v['img'] . '" data-alt="' . $v['alt'] . '" data-title="' . $v['title'] . '" ' . $selected . '>' . $v['title'] . '</option>';

					// Add a bean
					$x ++;
					$option_sr++;
				}

				// Close the <select> tag
				echo '</select>';

				// Some space
				echo '<br />';

				// Show the preview image.
				echo '<div>';

				// just in case.  You never know.
				if ( ! isset( $arrNum ) ) {
					$this->value = '';
				}

				// Set the default image.  To get the url from the default name,
				// we save the array count from the for/each loop, when the default image
				// is mark as selected.  Since the for/each loop starts at one, we must
				// substract one from the saved array number.  We then pull the url
				// out of the options array, and there we go.
				if ( '' == $this->value ) {
					echo '<img src="#" class="redux-preview-image" style="visibility:hidden;" id="image_' . $this->field['id'] . '">';
				} else {
					echo '<img src=' . $this->field['options'][ $this->value ]['img'] . ' class="redux-preview-image" id="image_' . $this->field['id'] . '">';
				}

				// Close the <div> tag.
				echo '</div>';
			} else {

				// No options specified.  Really?
				echo '<strong>' . esc_html__( 'No items of this type were found.', 'pgs-core' ) . '</strong>';
			}
		} //function

		/**
		 * Enqueue Function.
		 * If this field requires any scripts, or css define this function and register/enqueue the scripts/css
		 *
		 * @since ReduxFramework 1.0.0
		 */
		function enqueue() {

			$min = Redux_Functions::isMin();

			wp_enqueue_style( 'select2-css' );

			wp_enqueue_script(
				'field-select-image-new-js',
				$this->extension_url . '/field_select_image_new' . $min . '.js',
				array( 'jquery', 'select2-js', 'redux-js' ),
				time(),
				true
			);

			if ( $this->parent->args['dev_mode'] ) {
				wp_enqueue_style(
					'redux-field-select-image-new-css',
					$this->extension_url . '/field_select_image_new' . $min . '.css',
					array(),
					time(),
					'all'
				);
			}
		} //function
	} //class
}
