<?php
class PGS_Mailchimp {
	public $api_key          = '';
	public $list_id          = '';
	public $api_url          = 'https://server.api.mailchimp.com/3.0/';
	public $subscribe_status = 'subscribed';

	/**
	 * Constructor. Don't call directly, @see instance() instead.
	 *
	 * @see instance()
	 *
	 * @return void
	 **/
	public function __construct( $api_key = null, $list_id = null ) {
		$this->set_api_key( $api_key );
		$this->set_list_id( $list_id );
		$this->set_api_url();
	}

	// MailChimp API Key
	public function set_api_key( $api_key ) {
		$this->api_key = $api_key;
	}

	public function set_list_id( $list_id ) {
		$this->list_id = $list_id;
	}

	public function set_api_url() {
		$server = 'us1';

		if ( strstr( $this->api_key, '-' ) ) {
			list( $key, $server ) = explode( '-', $this->api_key, 2 );
			if ( ! $server ) {
				$server = "us1";
			}
		}

		$this->api_url = str_replace( 'server', $server, $this->api_url );
		$this->api_url = rtrim( $this->api_url, '/' ) . '/';
	}

	public function current_user_can() {
		$current_user_can = current_user_can( 'manage_options' );
		return apply_filters( 'pgs_mailchimp_current_user_can', $current_user_can );
	}

	public function get_messages( $msg_type = null ) {
		$message  = esc_html__( 'Something went wrong. Please try again later.', 'pgs-core' );
		$messages = array(
			'email_empty'             => esc_html__( 'Please enter email.', 'pgs-core' ),
			'something_wrong'         => esc_html__( 'Something went wrong. Please try again later.', 'pgs-core' ),
			'missing_api_key_list_id' => esc_html__( 'API Key & List ID is missing.', 'pgs-core' ),
			'missing_api_key'         => esc_html__( 'API Key is missing.', 'pgs-core' ),
			'missing_list_id'         => esc_html__( 'List ID is missing.', 'pgs-core' ),
			'invalid_email'           => esc_html__( 'Please enter a valid email.', 'pgs-core' ),
			'subscribed'              => esc_html__( 'Successfully Subscribed.', 'pgs-core' ),
			'subscription_pending'    => esc_html__( 'Successfully Subscribed. Please check confirmation email.', 'pgs-core' ),
		);
		if ( $msg_type && isset( $messages[ $msg_type ] ) && ! empty( $messages[ $msg_type ] ) ) {
			$message = $messages[ $msg_type ];
		}
		return $message;
	}

	public function subscribe( $email ) {
		if ( empty( $email ) ) {
			return new WP_Error( 'email_empty', $this->get_messages( 'email_empty' ) );
		}
		if ( empty( $this->api_key ) || empty( $this->list_id ) ) {
			if ( empty( $this->api_key ) && empty( $this->list_id ) ) {
				return new WP_Error( 'missing_api_key_list_id', $this->get_messages( 'missing_api_key_list_id' ) );
			} elseif ( empty( $this->api_key ) ) {
				return new WP_Error( 'missing_api_key', $this->get_messages( 'missing_api_key' ) );
			}elseif ( empty( $this->list_id ) ) {
				return new WP_Error( 'missing_list_id', $this->get_messages( 'missing_list_id' ) );
			} else {
				return new WP_Error( 'unknown_error', $this->get_messages() );
			}
		} else {
			$email_sanitized = filter_var( $email, FILTER_SANITIZE_EMAIL );
			if ( ! filter_var( $email_sanitized, FILTER_VALIDATE_EMAIL ) ) {
				return new WP_Error( 'invalid_email', $this->get_messages( 'invalid_email' ) );
			} else {
				$language     = $this->get_current_language_code();
				$language     = 'fr';
				$request_url  = trailingslashit( $this->api_url ) . "lists/{$this->list_id}/members/";
				$request_args = array(
					'headers' => array(
						'Authorization'   => 'Basic ' . base64_encode( 'user:' . $this->api_key ),
						'Content-Type'    => 'application/json',
					),
					'body' => json_encode(
						array(
							'email_address' => $email_sanitized,
							'status'        => 'subscribed',
							'language'      => $language,
						)
					),
				);

				$response = wp_remote_post( $request_url, $request_args );

				if ( is_wp_error( $response ) ) {
					// Handle WP error
					$error_message = $response->get_error_message();
					return new WP_Error( 'wp_remote_post_error', $error_message );
				} else {
					$response_code = wp_remote_retrieve_response_code($response);

					if ( $response_code === 200 || $response_code === 204 ) {
						// Email added successfully.
						return $this->get_messages( 'subscribed' );
					} else {

						// Handle API error
						$response_body = json_decode( wp_remote_retrieve_body( $response ), true );
						$error_title   = isset( $response_body['title'] ) ? $response_body['title'] : esc_html__( 'Error', 'pgs-core' );
						$error_message = isset( $response_body['detail'] ) ? $response_body['detail'] : $this->get_messages( 'something_wrong' );

						$mailchimp_replaces = array(
							' Use PUT to insert or update list members.' => '',
						);

						$mailchimp_replaces_search  = array_keys( $mailchimp_replaces );
						$mailchimp_replaces_replace = array_values( $mailchimp_replaces );
						$error_message              = str_replace( $mailchimp_replaces_search, $mailchimp_replaces_replace, $error_message );

						$message = sprintf( '%s: %s', $error_title, $error_message );
						return new WP_Error( 'mailchimp_error', $message );
					}
				}

				return $email_sanitized;
			}
		}
	}

	public function get_current_language_code() {
		$language_code = '';

		// Check if WPML is active
		if ( function_exists( 'icl_object_id' ) ) {
			global $sitepress;
			$language_code = $sitepress->get_current_language();
		} else {
			// Retrieve language code using WordPress default method
			$language_code = get_locale();
		}

		return $language_code;
	}
}
