<?php // phpcs:ignore WordPress.Files.FileName.NotHyphenatedLowercase
/**
 * Theme function
 *
 * @package CiyaShop
 */

/**
 * Add action hook to generate color customize.
 */
add_action( 'init', 'ciyashop_version_generate_color_customize_css', 99 );

if ( ! function_exists( 'ciyashop_version_generate_color_customize_css' ) ) {
	/**
	 * Generate color customize css.
	 */
	function ciyashop_version_generate_color_customize_css() {
		global $ciyashop_options;

		if ( ! class_exists( 'Redux' ) ) {
			return;
		}

		$header_schema        = '';
		$upload               = wp_upload_dir();
		$color_customize_file = $upload['basedir'] . '/ciyashop/color_customize.css';

		if ( ! file_exists( $color_customize_file ) ) {
			update_option( 'theme_version', false );
		}

		$theme_version = get_option( 'theme_version', false );

		if ( ! version_compare( THEME_VERSION, $theme_version ) ) {
			return;
		}

		// Generate the color customizer CSS.
		$primary_color   = isset( $ciyashop_options['primary_color'] ) ? esc_html( $ciyashop_options['primary_color'] ) : '';
		$secondary_color = isset( $ciyashop_options['secondary_color'] ) ? esc_html( $ciyashop_options['secondary_color'] ) : '';
		$tertiary_color  = isset( $ciyashop_options['tertiary_color'] ) ? esc_html( $ciyashop_options['tertiary_color'] ) : '';

		if ( isset( $ciyashop_options['header_type_select'] ) && 'predefined' !== $ciyashop_options['header_type_select'] ) {
			$header_schema = ciyashop_get_custom_header_schema();
		}

		$color_customize = ciyashop_get_color_scheme_colors( $primary_color, $secondary_color, $tertiary_color, $header_schema );
		ciyashop_generate_color_customize_css( $color_customize );

		update_option( 'theme_version', THEME_VERSION );
	}
}


/**
 * Add custom image sizes attribute to enhance responsive image functionality
 * for post thumbnails
 *
 * @param array $attr attributes.
 * @param array $attachment attachment.
 * @param int   $size size.
 */
function ciyashop_post_thumbnail_sizes_attr( $attr, $attachment, $size ) {
	if ( 'post-thumbnails' === $size ) {
		is_active_sidebar( 'sidebar-1' ) && $attr['sizes']   = '(max-width: 709px) 85vw, (max-width: 909px) 67vw, (max-width: 984px) 60vw, (max-width: 1362px) 62vw, 840px'; // phpcs:ignore Squiz.PHP.DisallowMultipleAssignments.Found
		! is_active_sidebar( 'sidebar-1' ) && $attr['sizes'] = '(max-width: 709px) 85vw, (max-width: 909px) 67vw, (max-width: 1362px) 88vw, 1200px'; // phpcs:ignore Squiz.PHP.DisallowMultipleAssignments.Found
	}
	return $attr;
}
add_filter( 'wp_get_attachment_image_attributes', 'ciyashop_post_thumbnail_sizes_attr', 10, 3 );

/**
 * Layout
 */
function ciyashop_site_layout() {
	global $ciyashop_options;

	$site_layout = 'fullwidth';

	if ( isset( $ciyashop_options['site_layout'] ) && ! empty( $ciyashop_options['site_layout'] ) ) {
		$site_layout = $ciyashop_options['site_layout'];
	}

	/**
	 * Filter the layout of the site.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $site_layout         Site layout.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$site_layout = apply_filters( 'ciyashop_site_layout', $site_layout, $ciyashop_options );

	return $site_layout;
}

/**
 * Logo Settings
 */
function ciyashop_logo_type() {
	global $ciyashop_options;

	$logo_type = 'image';
	if ( isset( $ciyashop_options['logo_type'] ) && ! empty( $ciyashop_options['logo_type'] ) ) {
		$logo_type = $ciyashop_options['logo_type'];
	}

	/**
	 * Filter the type of logo.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $logo_type           Logo Type.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$logo_type = apply_filters( 'ciyashop_logo_type', $logo_type, $ciyashop_options );

	return $logo_type;
}

/**
 * Logo URL
 */
function ciyashop_logo_url() {
	global $ciyashop_options;

	$logo_url = get_parent_theme_file_uri( '/images/logo.png' );

	if ( isset( $ciyashop_options['site-logo'] ) && ! empty( $ciyashop_options['site-logo']['url'] ) ) {
		$logo_url = $ciyashop_options['site-logo']['url'];
	}

	// Replace with Mobile log if on mobile device.
	if ( wp_is_mobile() && ciyashop_mobile_logo_url() ) {
		$logo_url = ciyashop_mobile_logo_url();
	}

	/**
	 * Filter the url of logo.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $logo_url            Logo URL.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$logo_url = apply_filters( 'ciyashop_logo_url', $logo_url, $ciyashop_options );

	return $logo_url;
}

/**
 * Sticky Logo URL
 */
function ciyashop_sticky_logo_url() {
	global $ciyashop_options;

	$logo_url = get_parent_theme_file_uri( '/images/logo-sticky.png' );

	if ( isset( $ciyashop_options['sticky-logo'] ) && ! empty( $ciyashop_options['sticky-logo']['url'] ) ) {
		$logo_url = $ciyashop_options['sticky-logo']['url'];
	}

	/**
	 * Filter the url of sticky logo.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $logo_url            Sticky logo URL.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$logo_url = apply_filters( 'ciyashop_sticky_logo_url', $logo_url, $ciyashop_options );

	return $logo_url;
}

/**
 * Mobile Logo URL
 */
function ciyashop_mobile_logo_url() {
	global $ciyashop_options;

	$logo_url = false;

	if ( isset( $ciyashop_options['mobile-logo'] ) && ! empty( $ciyashop_options['mobile-logo']['url'] ) ) {
		$logo_url = $ciyashop_options['mobile-logo']['url'];
	}

	/**
	 * Filter the url of mobile logo.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $logo_url            Mobile logo URL.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$logo_url = apply_filters( 'ciyashop_mobile_logo_url', $logo_url, $ciyashop_options );

	return $logo_url;
}


if ( ! function_exists( 'ciyashop_get_site_mobile_logo' ) ) {
	/**
	 * Site Mobile logo settings
	 */
	function ciyashop_get_site_mobile_logo() {
		global $ciyashop_options;

		if ( isset( $ciyashop_options['logo_type'] ) && 'image' === $ciyashop_options['logo_type'] && isset( $ciyashop_options['mobile-logo'] ) && isset( $ciyashop_options['mobile-logo']['url'] ) ) {
			return $ciyashop_options['mobile-logo']['url'];
		}

		return false;
	}
}

if ( ! function_exists( 'ciyashop_get_site_sticky_logo' ) ) {
	/**
	 * Site sticky logo settings
	 */
	function ciyashop_get_site_sticky_logo() {
		global $ciyashop_options;

		if ( isset( $ciyashop_options['logo_type'] ) && 'image' === $ciyashop_options['logo_type'] && isset( $ciyashop_options['sticky-logo'] ) && isset( $ciyashop_options['sticky-logo']['url'] ) ) {
			return $ciyashop_options['sticky-logo']['url'];
		}

		return false;
	}
}

/**
 * Site multi lang settings
 */
function ciyashop_get_multi_lang() {
	global $ciyashop_options;

	/**
	 * Checl WPML sitepress multilingual plugin activate
	 */
	if ( ciyashop_check_plugin_active( 'sitepress-multilingual-cms/sitepress.php' ) && function_exists( 'icl_get_languages' ) ) {
		$ls_settings = get_option( 'icl_sitepress_settings' );
		$languages   = icl_get_languages();
		if ( ! empty( $languages ) ) {
			?>
			<div class="language" id="drop">
				<?php
				/* Display Current language */
				foreach ( $languages as $k => $al ) {
					if ( 1 === (int) $al['active'] ) {
						$flag_url_stat = ciyashop_wpml_custom_flag_exists( $al['country_flag_url'] );
						?>
						<a href="#">
							<?php
							if ( $al['country_flag_url'] && $flag_url_stat ) {
								?>
								<img src="<?php echo esc_url( $al['country_flag_url'] ); ?>" height="12" alt="<?php echo esc_attr( $al['language_code'] ); ?>" width="18" />&nbsp;
								<?php
							}
							echo icl_disp_language( $al['native_name'], $al['translated_name'] ) . '&nbsp;<i class="fa fa-angle-down">&nbsp;</i>'; // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotE
							?>
						</a>
						<?php
						unset( $languages[ $k ] );
						break;
					}
				}
				?>
				<ul class="drop-content">
					<?php
					foreach ( $languages as $l ) {
						if ( ! $l['active'] ) {
							$flag_url_stat = ciyashop_wpml_custom_flag_exists( $l['country_flag_url'] );
							?>
							<li>
								<a href="<?php echo esc_url( $l['url'] ); ?>">
									<?php
									if ( $l['country_flag_url'] && $flag_url_stat ) {
										?>
										<img src="<?php echo esc_url( $l['country_flag_url'] ); ?>" height="12" alt="<?php echo esc_attr( $l['language_code'] ); ?>" width="18" />
										<?php
									}
									if ( isset( $ls_settings['icl_lso_flags'] ) && 1 === (int) $ls_settings['icl_lso_flags'] ) {
										echo icl_disp_language( $l['native_name'], $l['translated_name'] ); // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotE
									} else {
										echo icl_disp_language( $l['native_name'] ); // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotE
									}
									?>
								</a>
							</li>
							<?php
						}
					}
					?>
				</ul>
			</div>
			<?php
		}
	}
}

/**
 * Site login options settings
 *
 * @param string   $items items.
 * @param stdClass $args arguments.
 */
function ciyashop_get_login_options( $items, $args ) {
	/**
	 * Filters active plugins.
	 *
	 * @param array $active_plugins List of active plugins.
	 *
	 * @visible false
	 * @ignore
	 */
	if ( function_exists( 'WC' ) ) {
		if ( ! is_user_logged_in() && 'top_menu' === $args->theme_location ) {
			$items .= '<li><a href="javascript:void(0);" data-toggle="modal" data-target="#pgs_login_form"><i class="fa fa-sign-in">&nbsp;</i> ' . esc_html__( 'Login', 'ciyashop' ) . '</a></li>';
		} elseif ( is_user_logged_in() && 'top_menu' === $args->theme_location ) {
			$items .= '<li><a href="' . esc_url( wp_logout_url( home_url() ) ) . '" title="' . esc_attr__( 'Logout', 'ciyashop' ) . '" class="logout"><i class="fa fa-sign-out">&nbsp;</i>  ' . esc_html__( 'Logout', 'ciyashop' ) . '</a></li>';
		}
	}
	return $items;
}
add_filter( 'wp_nav_menu_items', 'ciyashop_get_login_options', 10, 2 );

/**
 * Ciyashop footer copyright
 */
function ciyashop_footer_copyright() {
	global $ciyashop_options;

	$allowed_tags = wp_kses_allowed_html( 'post' );

	$footer_copyright = sprintf(
		esc_html__( 'Copyright', 'ciyashop' ) . ' &copy; <span class="copy_year">%1$s</span>, <a href="%2$s" title="%3$s">%4$s</a> ' . esc_html__( 'All Rights Reserved.', 'ciyashop' ),
		gmdate( 'Y' ),
		esc_url( home_url( '/' ) ),
		esc_attr( get_bloginfo( 'name', 'display' ) ),
		esc_html( get_bloginfo( 'name', 'display' ) )
	);

	if ( isset( $ciyashop_options['footer_copyright'] ) && ! empty( $ciyashop_options['footer_copyright'] ) ) {
		$footer_copyright = $ciyashop_options['footer_copyright'];
		$footer_copyright = do_shortcode( $footer_copyright );
	}

	/**
	 * Filter the footer copyright content.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $footer_copyright    Footer copyright content.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$footer_copyright = apply_filters( 'ciyashop_footer_copyright', $footer_copyright, $ciyashop_options );

	/**
	 * Filter the content before copyright content.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $before_copyright    Before copyright content. Default null.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$footer_copyright_before = apply_filters( 'ciyashop_footer_copyright_before', '' );

	/**
	 * Filter the content after copyright content.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $after_copyright     After copyright content. Default null.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$footer_copyright_after = apply_filters( 'ciyashop_footer_copyright_after', '' );

	$footer_copyright = $footer_copyright_before . $footer_copyright . $footer_copyright_after;

	echo wp_kses(
		$footer_copyright,
		array(
			'a'      => $allowed_tags['a'],
			'br'     => $allowed_tags['br'],
			'em'     => $allowed_tags['em'],
			'strong' => $allowed_tags['strong'],
			'span'   => $allowed_tags['span'],
			'div'    => $allowed_tags['div'],
		)
	);
}

/**
 * Footer Creadits
 */
function ciyashop_footer_credits() {
	global $ciyashop_options;

	$ciyashop_credits = sprintf(
		// Translators: %s is the theme credit link.
		esc_html__( 'Developed and designed by %s', 'ciyashop' ),
		'<a href="' . esc_url( 'http://www.potenzaglobalsolutions.com/' ) . '">' . esc_html__( 'Potenza Global Solutions', 'ciyashop' ) . '</a>'
	);

	/**
	 * Filter the footer credit content.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $ciyashop_credits    Credit content.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$ciyashop_credits = apply_filters( 'ciyashop_credits', $ciyashop_credits, $ciyashop_options );

	echo wp_kses( $ciyashop_credits, ciyashop_allowed_html( array( 'i', 'a', 'span', 'div', 'ul', 'ol', 'li' ) ) );
}

/**
 * Get product listing page sidebar settings.
 */
function ciyashop_shop_page_sidebar() {
	global $ciyashop_options;

	$shop_sidebar = ( isset( $ciyashop_options['shop_sidebar'] ) && ! empty( $ciyashop_options['shop_sidebar'] ) ) ? $ciyashop_options['shop_sidebar'] : 'left';

	return $shop_sidebar;
}

/**
 * Get product page sidebar settings.
 */
function ciyashop_product_page_sidebar() {
	global $ciyashop_options;

	$product_sidebar = ( isset( $ciyashop_options['product-page-sidebar'] ) && ! empty( $ciyashop_options['product-page-sidebar'] ) ) ? $ciyashop_options['product-page-sidebar'] : 'left';

	return $product_sidebar;
}

/**
 * Get sidebar on Mobile.
 */
function ciyashop_show_sidebar_on_mobile() {
	global $ciyashop_options;

	$show_sidebar_on_mobile = true;

	if ( ! $ciyashop_options || ! isset( $ciyashop_options['show_sidebar_on_mobile'] ) || is_null( $ciyashop_options['show_sidebar_on_mobile'] ) ) {
		return $show_sidebar_on_mobile;
	}

	return (bool) $ciyashop_options['show_sidebar_on_mobile'];
}

/**
 * Get sidebar position on Mobile.
 */
function ciyashop_mobile_sidebar_position() {
	global $ciyashop_options;

	if ( isset( $ciyashop_options['shop_sidebar_position_mobile'] ) && ! empty( $ciyashop_options['shop_sidebar_position_mobile'] ) ) {
		return $ciyashop_options['shop_sidebar_position_mobile'];
	}
	return 'bottom';
}
/**
 * Ciyashop device type
 */
function ciyashop_device_type() {
	return wp_is_mobile() ? 'mobile' : 'desktop';
}

// Change number or products per row to 3.
add_filter( 'loop_shop_columns', 'ciyashop_loop_columns' );
if ( ! function_exists( 'ciyashop_loop_columns' ) ) {
	/**
	 * Ciyashop loop columns
	 */
	function ciyashop_loop_columns() {
		global $ciyashop_options;
		$pro_col_sel = 4;
		if ( isset( $ciyashop_options['pro-col-sel'] ) && ! empty( $ciyashop_options['pro-col-sel'] ) ) {
			$pro_col_sel = $ciyashop_options['pro-col-sel'];
		}
		return $pro_col_sel; // 3 products per row
	}
}

add_filter( 'bcn_breadcrumb_title', 'ciyashop_bcn_breadcrumb_title', 10, 3 );
/**
 * Change site name to home in breadcrumb
 *
 * @param string $title .
 * @param string $type .
 * @param string $id .
 */
function ciyashop_bcn_breadcrumb_title( $title, $type, $id ) {
	if ( 'home' === $type[0] ) {
		$title = esc_html__( 'Home', 'ciyashop' );
	}
	return $title;
}
/**
 * Class builder
 *
 * @param string $class .
 */
function ciyashop_class_builder( $class = '' ) {
	$classes = array();

	if ( $class ) {
		if ( ! is_array( $class ) ) {
			$class = preg_split( '#\s+#', $class );
		}
		$classes = array_map( 'esc_attr', $class );
	} else {
		// Ensure that we always coerce class to being an array.
		$class = array();
	}
	$classes = array_map( 'esc_attr', $classes );

	return implode( ' ', array_filter( array_unique( $classes ) ) );
}
/**
 * Header type
 */
function ciyashop_header_type() {

	global $ciyashop_options;

	if ( isset( $ciyashop_options['header_type_select'] ) && 'header_builder' === $ciyashop_options['header_type_select'] ) {
		$ciyashop_header_type = 'custom';
	} else {
		$ciyashop_header_type = ( isset( $ciyashop_options['header_type'] ) && $ciyashop_options['header_type'] ) ? $ciyashop_options['header_type'] : 'menu-center';
	}

	/**
	 * Filter header_type.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $header_type         Header Type.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$ciyashop_header_type = apply_filters( 'ciyashop_header_type', $ciyashop_header_type, $ciyashop_options );

	return $ciyashop_header_type;
}

if ( ! function_exists( 'ciyashop_footer_type' ) ) {
	/**
	 * Get Footer Type
	 *
	 * @since  1.0.0
	 *
	 * @return string
	 */
	function ciyashop_footer_type() {

		global $ciyashop_options, $post;

		$ciyashop_footer_type = ( ! empty( $ciyashop_options['footer_type_select'] ) ) ? $ciyashop_options['footer_type_select'] : 'default';

		/**
		 * Filter header_type_select.
		 *
		 * @since 1.0.0
		 *
		 * @param string $ciyashop_footer_type Footer Type.
		 * @param array $ciyashop_options An array of theme options.
		 *
		 * @visible true
		 */
		$ciyashop_footer_type = apply_filters( 'ciyashop_footer_type', $ciyashop_footer_type, $ciyashop_options );

		return $ciyashop_footer_type;
	}
}

if ( ! function_exists( 'ciyashop_get_custom_footer_data' ) ) {
	/**
	 * Get Footer data
	 *
	 * @since  1.0.0
	 *
	 * @param  int $footer_id footer id.
	 * @return array
	 */
	function ciyashop_get_custom_footer_data( $footer_id = '' ) {
		global $ciyashop_options, $wpdb;

		if ( ! $footer_id ) {
			if ( function_exists( 'ciyashop_footer_type' ) && ciyashop_footer_type() ) {
				if ( 'footer_builder' === ciyashop_footer_type() ) {
					if ( isset( $ciyashop_options['custom_footers'] ) ) {
						if ( $ciyashop_options['custom_footers'] ) {
							$footer_id = $ciyashop_options['custom_footers'];
						} else {
							return false;
						}
					}
				}
			}

			if ( ! $footer_id ) {
				return false;
			}
		}

		$footer_layout_data = $wpdb->get_results(
			$wpdb->prepare(
				'
				SELECT * FROM ' . $wpdb->prefix . 'cs_footer_builder
				WHERE id = %d
				',
				$footer_id
			)
		);

		if ( ! $footer_layout_data ) {
			return false;
		}
		$footer_data = unserialize( $footer_layout_data[0]->value );

		return $footer_data;
	}
}

/**
 * Categories menu status
 */
function ciyashop_categories_menu_status() {

	global $ciyashop_options;

	if ( isset( $ciyashop_options['categories_menu_status'] ) && ! empty( $ciyashop_options['categories_menu_status'] ) ) {
		$ciyashop_header_type   = ciyashop_header_type();
		$categories_menu_status = in_array( $ciyashop_header_type, array( 'default', 'logo-center', 'topbar-with-main-header' ), true ) ? $ciyashop_options['categories_menu_status'] : 'disable';
	} else {
		$categories_menu_status = 'disable';
	}

	/**
	 * Filters the status of the categories menu.
	 *
	 * @since 1.0.0
	 *
	 * @param     string       $menu_status         Categories menu status.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$categories_menu_status = apply_filters( 'ciyashop_categories_menu_status', $categories_menu_status, $ciyashop_options );

	return $categories_menu_status;
}
/**
 * Topbar enable
 */
function ciyashop_topbar_enable() {

	global $ciyashop_options;

	$topbar_enable_stat = ( isset( $ciyashop_options['topbar_enable'] ) ) ? $ciyashop_options['topbar_enable'] : false;

	if ( $topbar_enable_stat ) {
		$topbar_enable = 'enable';
	} else {
		$topbar_enable = 'disable';
	}

	/**
	 * Filters whether to enable topbar.
	 *
	 * @since 1.0.0
	 *
	 * @param     boolean      $topbar_enable       Whether to enable the topbar.
	 * @param     array        $ciyashop_options    An array of theme options.
	 *
	 * @visible true
	 */
	$topbar_enable = apply_filters( 'ciyashop_topbar_enable', $topbar_enable, $ciyashop_options );

	return $topbar_enable;
}
/**
 * Topbar mobile enable
 */
function ciyashop_topbar_mobile_enable() {

	global $ciyashop_options;

	$topbar_mobile_enable_stat = ( isset( $ciyashop_options['topbar_mobile_enable'] ) ) ? $ciyashop_options['topbar_mobile_enable'] : true;

	if ( $topbar_mobile_enable_stat && 'enable' === ciyashop_topbar_enable() ) {
		$topbar_mobile_enable = 'enable';
	} else {
		$topbar_mobile_enable = 'disable';
	}

	/**
	 * Filters whether to enable topbar in mobile.
	 *
	 * @since 1.0.0
	 *
	 * @param     string      $enable_topbar        Whether to enable topbar in mobile.
	 * @param     array       $ciyashop_options     An array of theme options.
	 *
	 * @visible true
	 */
	$topbar_mobile_enable = apply_filters( 'ciyashop_topbar_mobile_enable', $topbar_mobile_enable, $ciyashop_options );

	return $topbar_mobile_enable;
}
/**
 * Social profiles
 */
function ciyashop_social_profiles() {
	global $ciyashop_options;

	/**
	 * Filters the list of social profiles.
	 *
	 * @param    array    $social_profiles        An array of social profiles.
	 *
	 * @visible true
	 */
	$social_profiles = apply_filters(
		'ciyashop_social_profiles',
		array(
			'facebook'   => array(
				'key'  => 'facebook',
				'name' => esc_html__( 'Facebook', 'ciyashop' ),
				'icon' => '<i class="fa fa-facebook"></i>',
			),
			'twitter'    => array(
				'key'  => 'twitter',
				'name' => esc_html__( 'Twitter', 'ciyashop' ),
				'icon' => '<i class="fa fa-twitter"></i>',
			),
			'googleplus' => array(
				'key'  => 'googleplus',
				'name' => esc_html__( 'Google+', 'ciyashop' ),
				'icon' => '<i class="fa fa-google-plus"></i>',
			),
			'dribbble'   => array(
				'key'  => 'Dribbble',
				'name' => esc_html__( 'Dribbble', 'ciyashop' ),
				'icon' => '<i class="fa fa-dribbble"></i>',
			),
			'vimeo'      => array(
				'key'  => 'vimeo',
				'name' => esc_html__( 'Vimeo', 'ciyashop' ),
				'icon' => '<i class="fa fa-vimeo"></i>',
			),
			'pinterest'  => array(
				'key'  => 'pinterest',
				'name' => esc_html__( 'Pinterest', 'ciyashop' ),
				'icon' => '<i class="fa fa-pinterest"></i>',
			),
			'behance'    => array(
				'key'  => 'behance',
				'name' => esc_html__( 'Behance', 'ciyashop' ),
				'icon' => '<i class="fa fa-behance"></i>',
			),
			'linkedin'   => array(
				'key'  => 'linkedin',
				'name' => esc_html__( 'Linkedin', 'ciyashop' ),
				'icon' => '<i class="fa fa-linkedin"></i>',
			),
			'youtube'    => array(
				'key'  => 'youtube',
				'name' => esc_html__( 'YouTube', 'ciyashop' ),
				'icon' => '<i class="fa fa-youtube-play"></i>',
			),
			'instagram'  => array(
				'key'  => 'instagram',
				'name' => esc_html__( 'instagram', 'ciyashop' ),
				'icon' => '<i class="fa fa-instagram"></i>',
			),
		)
	);

	if ( ! isset( $ciyashop_options['social_media_icons'] ) || empty( $ciyashop_options['social_media_icons'] ) ) {
		return '';
	}
	$total_social_icon = count( $ciyashop_options['social_media_icons']['redux_repeater_data'] );

	$social_profiles_data = array();
	for ( $i = 0; $i < $total_social_icon; $i++ ) {
		if ( isset( $ciyashop_options['social_media_icons'] ) && is_array( $ciyashop_options['social_media_icons'] ) && 'custom' === $ciyashop_options['social_media_icons']['social_media_type'][ $i ] ) {
			$social_icon_href = $ciyashop_options['social_media_icons']['social_media_url'][ $i ];
			$social_icon      = $ciyashop_options['social_media_icons']['custom_soical_icon'][ $i ];
			$social_title     = $ciyashop_options['social_media_icons']['custom_social_title'][ $i ];

			if ( ! empty( $social_icon_href ) && ! empty( $social_icon ) ) {
				if ( empty( array_intersect( explode( ' ', $social_icon ), array( 'far', 'fa-regular', 'fas', 'fa-solid', 'fab', 'fa-brands' ) ) ) ) {
					$social_icon = "fa {$social_icon}";
				}

				$social_profiles_data[] = array(
					'link'  => esc_url( $social_icon_href ),
					'icon'  => '<i class="' . esc_attr( $social_icon ) . '"></i>',
					'title' => esc_attr( $social_title ),
				);
			}
		} else {
			$social_icon_href = isset( $ciyashop_options['social_media_icons']['social_media_url'][ $i ] ) ? $ciyashop_options['social_media_icons']['social_media_url'][ $i ] : '';
			$social_icon      = $ciyashop_options['social_media_icons']['social_media_type'][ $i ];
			$social_title     = ucfirst( $ciyashop_options['social_media_icons']['social_media_type'][ $i ] );

			if ( ! empty( $social_icon_href ) && isset( $social_profiles[ $social_icon ]['icon'] ) ) {
				$social_profiles_data[] = array(
					'link'  => esc_url( $social_icon_href ),
					'icon'  => $social_profiles[ $social_icon ]['icon'],
					'title' => esc_attr( $social_title ),
				);
			}
		}
	}
	return $social_profiles_data;
}

/**
 * Opening hours
 */
function ciyashop_opening_hours() {
	global $ciyashop_options;

	$ciyashop_opening_hours_data = array(

		esc_html__( 'Monday', 'ciyashop' )    => isset( $ciyashop_options['mon_time'] ) ? $ciyashop_options['mon_time'] : '',
		esc_html__( 'Tuesday', 'ciyashop' )   => isset( $ciyashop_options['tue_time'] ) ? $ciyashop_options['tue_time'] : '',
		esc_html__( 'Wednesday', 'ciyashop' ) => isset( $ciyashop_options['wed_time'] ) ? $ciyashop_options['wed_time'] : '',
		esc_html__( 'Thursday', 'ciyashop' )  => isset( $ciyashop_options['thu_time'] ) ? $ciyashop_options['thu_time'] : '',
		esc_html__( 'Friday', 'ciyashop' )    => isset( $ciyashop_options['fri_time'] ) ? $ciyashop_options['fri_time'] : '',
		esc_html__( 'Saturday', 'ciyashop' )  => isset( $ciyashop_options['sat_time'] ) ? $ciyashop_options['sat_time'] : '',
		esc_html__( 'Sunday', 'ciyashop' )    => isset( $ciyashop_options['sun_time'] ) ? $ciyashop_options['sun_time'] : '',

	);
	return $ciyashop_opening_hours_data;
}

/**
 * Return whether Visual Composer is enabled on a page/post or not.
 *
 * @param string $post_id .
 * $post_id = numeric post_id.
 * return true/false.
 */
function ciyashop_is_vc_enabled( $post_id = '' ) {
	global $post;

	if ( ! empty( $post_id ) ) {
		$post = get_post( $post_id ); // phpcs:ignore WordPress.WP.GlobalVariablesOverride
	}

	if ( is_search() || ( empty( $post ) ) ) {
		return;
	}

	if ( empty( $post_id ) ) {
		$post_id = $post->ID;
	}
	$vc_enabled = get_post_meta( $post_id, '_wpb_vc_js_status', true );
	return ( 'true' === $vc_enabled ) ? true : false;
}

/**
 * Converts a multidimensional array of CSS rules into a CSS string.
 *
 * @param array $rules .
 * @param int   $indent .
 *   An array of CSS rules in the form of:
 *   array('selector'=>array('property' => 'value')). Also supports selector
 *   nesting, e.g.,
 *   array('selector' => array('selector'=>array('property' => 'value'))).
 *
 * @return string
 *   A CSS string of rules. This is not wrapped in style tags.
 *
 * source : http://www.grasmash.com/article/convert-nested-php-array-css-string
 */
function ciyashop_generate_css_properties( $rules, $indent = 0 ) {
	$css    = '';
	$prefix = str_repeat( '  ', $indent );
	foreach ( $rules as $key => $value ) {
		if ( is_array( $value ) ) {
			$selector   = $key;
			$properties = $value;

			$css .= $prefix . "$selector {\n";
			$css .= $prefix . ciyashop_generate_css_properties( $properties, $indent + 1 );
			$css .= $prefix . "}\n";
		} else {
			$property = $key;
			$css     .= $prefix . "$property: $value;\n";
		}
	}
	return $css;
}

add_action( 'wp_footer', 'ciyashop_promo_popup' );
/**
 * Promo popup
 */
function ciyashop_promo_popup() {
	global $ciyashop_options;
	$content_css = '';

	if ( isset( $ciyashop_options['promo_popup'] ) && $ciyashop_options['promo_popup'] ) {

		$content_style = array();

		// Background.
		if ( isset( $ciyashop_options['popup-background']['background-color'] ) && ! empty( $ciyashop_options['popup-background']['background-color'] ) && 'transparent' !== $ciyashop_options['popup-background']['background-color'] ) {
			$content_style['background-color'] = 'background-color:' . $ciyashop_options['popup-background']['background-color'];
		}
		if ( isset( $ciyashop_options['popup-background']['background-repeat'] ) && ! empty( $ciyashop_options['popup-background']['background-repeat'] ) ) {
			$content_style['background-repeat'] = 'background-repeat:' . $ciyashop_options['popup-background']['background-repeat'];
		}
		if ( isset( $ciyashop_options['popup-background']['background-size'] ) && ! empty( $ciyashop_options['popup-background']['background-size'] ) ) {
			$content_style['background-size'] = 'background-size:' . $ciyashop_options['popup-background']['background-size'];
		}
		if ( isset( $ciyashop_options['popup-background']['background-attachment'] ) && ! empty( $ciyashop_options['popup-background']['background-attachment'] ) ) {
			$content_style['background-attachment'] = 'background-attachment:' . $ciyashop_options['popup-background']['background-attachment'];
		}
		if ( isset( $ciyashop_options['popup-background']['background-position'] ) && ! empty( $ciyashop_options['popup-background']['background-position'] ) ) {
			$content_style['background-position'] = 'background-position:' . $ciyashop_options['popup-background']['background-position'];
		}
		if ( isset( $ciyashop_options['popup-background']['background-image'] ) && ! empty( $ciyashop_options['popup-background']['background-image'] ) ) {
			$content_style['background-image'] = 'background-image:url(' . $ciyashop_options['popup-background']['background-image'] . ')';
		}
		if ( $content_style && is_array( $content_style ) && ! empty( $content_style ) ) {
			$content_css = implode( ';', array_filter( array_unique( $content_style ) ) );
		}
		?>
		<div class="ciyashop-promo-popup mfp-hide">
			<div class="ciyashop-popup-inner" style="<?php echo esc_attr( $content_css ); ?>">
				<?php echo do_shortcode( $ciyashop_options['popup_text'] ); ?>
			</div>
		</div>
		<?php
	}
}

// Source : https://support.advancedcustomfields.com/forums/topic/color-picker-values/.
/**
 * Convert hexdec color string to rgb(a) string.
 *
 * @param string $color .
 * @param bool   $opacity .
 */
function ciyashop_hex2rgba( $color = '', $opacity = false ) {

	$default = 'rgb(0,0,0)';

	// Return default if no color provided.
	if ( empty( $color ) ) {
		return $default;
	}

	// Sanitize $color if "#" is provided.
	if ( '#' === (string) $color[0] ) {
		$color = substr( $color, 1 );
	}

	// Check if color has 6 or 3 characters and get values.
	if ( 6 === (int) strlen( $color ) ) {
		$hex = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
	} elseif ( 3 === (int) strlen( $color ) ) {
		$hex = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
	} else {
		return $default;
	}

	// Convert hexadec to rgb.
	$rgb = array_map( 'hexdec', $hex );

	// Check if opacity is set(rgba or rgb).
	if ( $opacity ) {
		if ( abs( $opacity ) > 1 ) {
			$opacity = 1.0;
		}
		$output = 'rgba(' . implode( ',', $rgb ) . ',' . $opacity . ')';
	} else {
		$output = 'rgb(' . implode( ',', $rgb ) . ')';
	}

	// Return rgb(a) color string.
	return $output;
}
/**
 * Hex2rgb
 *
 * @param string $color .
 */
function ciyashop_hex2rgb( $color ) {
	if ( '#' === (string) $color[0] ) {
		$color = substr( $color, 1 );
	}
	if ( 6 === (int) strlen( $color ) ) {
		list( $r, $g, $b ) = array( $color[0] . $color[1], $color[2] . $color[3], $color[4] . $color[5] );
	} elseif ( 3 === (int) strlen( $color ) ) {
		list( $r, $g, $b ) = array( $color[0] . $color[0], $color[1] . $color[1], $color[2] . $color[2] );
	} else {
		return false;
	}

	$r = hexdec( $r );
	$g = hexdec( $g );
	$b = hexdec( $b );

	return array(
		'r' => $r,
		'g' => $g,
		'b' => $b,
	);
}
/**
 * Sticky_header
 */
function ciyashop_sticky_header() {
	global $ciyashop_options;

	$sticky_header = isset( $ciyashop_options['sticky_header'] ) ? $ciyashop_options['sticky_header'] : true;

	/**
	 * Filters whether to enable sticky header.
	 *
	 * @since 1.0.0
	 *
	 * @param     boolean       $enable_sticky      Whether to enable the sticky header.
	 * @param     array         $ciyashop_options   An array of theme options.
	 *
	 * @visible true
	 */
	$sticky_header = apply_filters( 'ciyashop_sticky_header', $sticky_header, $ciyashop_options );

	return $sticky_header;
}
/**
 * Mobile_sticky_header
 */
function ciyashop_mobile_sticky_header() {
	global $ciyashop_options;

	$ciyashop_sticky_header = ciyashop_sticky_header();

	if ( ! $ciyashop_sticky_header ) {
		return false;
	}

	$mobile_sticky_header = isset( $ciyashop_options['mobile_sticky_header'] ) ? $ciyashop_options['mobile_sticky_header'] : true;

	/**
	 * Filters whether to enable sticky header in mobile.
	 *
	 * @since 1.0.0
	 *
	 * @param     boolean       $enable_sticky      Whether to enable the sticky header in mobile.
	 * @param     array         $ciyashop_options   An array of theme options.
	 *
	 * @visible true
	 */
	$mobile_sticky_header = apply_filters( 'ciyashop_mobile_sticky_header', $mobile_sticky_header, $ciyashop_options );

	return $mobile_sticky_header;
}

/**
 * Display Related Post in single blog post
 */
function ciyashop_related_posts() {
	global $ciyashop_options;

	$related_posts = ( isset( $ciyashop_options['related_posts'] ) ) ? $ciyashop_options['related_posts'] : false;

	if ( 0 === (int) $related_posts ) {
		return;
	}

	$category_terms = wp_get_post_terms( get_the_ID(), 'category' );
	$cat_id         = array();
	if ( ! empty( $category_terms ) ) {
		foreach ( $category_terms as $value ) {
			$cat_id[] = $value->term_id;
		}
	}
	$args               = array(
		'post_type'      => 'post',
		'posts_per_page' => 10,
		'post__not_in'   => array( get_the_ID() ),
		'tax_query'      => array(
			array(
				'taxonomy' => 'category',
				'field'    => 'id',
				'terms'    => $cat_id,
			),
		),
	);
	$related_post_query = new WP_Query( $args );

	$owl_options_args = array(
		'items'              => 3,
		'loop'               => false,
		'dots'               => false,
		'nav'                => true,
		'margin'             => 10,
		'autoplay'           => true,
		'autoplayHoverPause' => true,
		'smartSpeed'         => 1000,
		'responsive'         => array(
			0    => array(
				'items' => 1,
			),
			480  => array(
				'items' => 1,
			),
			577  => array(
				'items' => 2,
			),
			980  => array(
				'items' => 3,
			),
			1200 => array(
				'items' => 3,
			),
		),
		'navText'            => array(
			"<i class='fas fa-angle-left fa-2x'></i>",
			"<i class='fas fa-angle-right fa-2x'></i>",
		),
		'lazyLoad'           => ( isset( $ciyashop_options['enable_lazyload'] ) && $ciyashop_options['enable_lazyload'] ) ? true : false,
	);

	$owl_options = wp_json_encode( $owl_options_args );
	if ( $related_post_query->have_posts() ) {
		?>
		<div class="related-posts">
			<h3 class="title"><?php esc_html_e( 'Related Posts', 'ciyashop' ); ?></h3>
			<div class="related-posts-section">
				<div class="owl-carousel owl-theme owl-carousel-options" data-owl_options="<?php echo esc_attr( $owl_options ); ?>">
				<?php
				while ( $related_post_query->have_posts() ) {
					$related_post_query->the_post();
					?>
					<div class="item">
						<div class="related-post-info">
							<div class="post-image clearfix<?php echo ( ! has_post_thumbnail() ) ? ' no-post-image' : ''; ?>">
								<?php
								if ( has_post_thumbnail() ) {
									if ( isset( $ciyashop_options['enable_lazyload'] ) && $ciyashop_options['enable_lazyload'] ) {
										echo '<img class="owl-lazy img-fluid" src="' . esc_url( LOADER_IMAGE ) . '" data-src="' . esc_url( get_the_post_thumbnail_url( '', 'ciyashop-latest-post-thumbnail' ) ) . '" alt="' . esc_attr( get_the_title() ) . '" />';
									} else {
										echo '<img class="img-fluid" src="' . esc_url( get_the_post_thumbnail_url( '', 'ciyashop-latest-post-thumbnail' ) ) . '" alt="' . esc_attr( get_the_title() ) . '" />';
									}
								}
								?>
							</div>
							<span class="post-<?php echo ( ! has_post_thumbnail() ) ? 'no-image-text' : 'image-text'; ?>">
								<?php the_title( '<h5 class="title"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">', '</a></h5>' ); ?>
							</span>
						</div>
					</div>
					<?php
				}
				?>
				</div>
			</div>
		</div>
		<?php
		wp_reset_postdata();
	}
}

/**
 * Display Author box in single blog post
 */
function ciyashop_authorbox() {
	global $ciyashop_options;

	$author_details = ( isset( $ciyashop_options['author_details'] ) ) ? $ciyashop_options['author_details'] : true;

	if ( 0 === (int) $author_details ) {
		return;
	}

	if ( is_singular() && get_the_author_meta( 'description' ) ) : // If a user has filled out their description and this is a multi-author blog, show a bio on their entries.
		?>
		<div class="author-info">
			<div class="author-avatar media-left">
				<?php
				/**
				 * Filters avatar size of author.
				 *
				 * @since 1.0.0
				 *
				 * @param     int       $avatar_size      Avatar size. Default 68.
				 *
				 * @visible true
				 */
				$author_bio_avatar_size = apply_filters( 'ciyashop_author_bio_avatar_size', 68 );
				echo get_avatar( get_the_author_meta( 'user_email' ), $author_bio_avatar_size );
				?>
			</div><!-- .author-avatar -->
			<div class="author-description media-body">
				<h5 class="title author-title">
					<?php
					printf(
						/* translators: %s: Name of athor */
						esc_html__( 'About %s', 'ciyashop' ),
						get_the_author()
					);
					?>
				</h5>
				<p><?php the_author_meta( 'description' ); ?></p>
			</div><!-- .author-description -->
		</div><!-- .author-info -->
		<?php
	endif;
}
/**
 * Get excerpt max charlength
 *
 * @param string $charlength .
 * @param string $excerpt .
 */
function ciyashop_get_excerpt_max_charlength( $charlength, $excerpt = null ) {
	if ( empty( $excerpt ) ) {
		$excerpt = get_the_excerpt();
	}
	$charlength++;

	if ( mb_strlen( $excerpt ) > $charlength ) {
		$subex   = mb_substr( $excerpt, 0, $charlength - 5 );
		$exwords = explode( ' ', $subex );
		$excut   = - ( mb_strlen( $exwords[ count( $exwords ) - 1 ] ) );

		$new_excerpt = '';
		if ( $excut < 0 ) {
			$new_excerpt = mb_substr( $subex, 0, $excut );
		} else {
			$new_excerpt = $subex;
		}
		$new_excerpt .= '[...]';
		return $new_excerpt;
	}

	return $excerpt;
}
/**
 * The excerpt max charlength
 *
 * @param string $charlength .
 * @param string $excerpt .
 */
function ciyashop_the_excerpt_max_charlength( $charlength, $excerpt = null ) {
	$ciyashop_excerpt = ciyashop_get_excerpt_max_charlength( $charlength, $excerpt );
	echo wp_kses_post( $ciyashop_excerpt );
}
/**
 * Validate CSS
 *
 * @param string $str .
 * @param array  $units .
 */
function ciyashop_validate_css_unit( $str = '', $units = array() ) {

	// bail early if any param is empty.
	if ( ! is_string( $str ) || empty( $str ) || ! is_array( $units ) || empty( $units ) ) {
		return false;
	}

	// prepare units string.
	$units_str = implode( '|', $units );

	// prepare regex.
	$reg_ex = '/^(auto|0)$|^[+-]?[0-9]+.?([0-9]+)?(' . $units_str . ')$/';

	// check match.
	preg_match_all( $reg_ex, $str, $matches, PREG_SET_ORDER, 0 );

	// check if matched found.
	if ( count( $matches ) > 0 ) {
		return true;
	}
	return false;
}

add_action( 'customize_save_response', 'ciyashop_customize_save_response', 10, 2 );
/**
 * Update Theme option favicon icon from customizer site icon
 *
 * @param string $response .
 * @param string $data .
 */
function ciyashop_customize_save_response( $response, $data ) {
	global $ciyashop_globals;

	if ( isset( $_POST['customized'] ) ) {
		$post_values = json_decode( sanitize_text_field( wp_unslash( $_POST['customized'] ) ), true );

		if ( isset( $post_values['site_icon'] ) && ! empty( $post_values['site_icon'] ) ) {
			$opt_name         = $ciyashop_globals['theme_option'];
			$ciyashop_options = get_option( $opt_name );
			$site_icon        = $post_values['site_icon'];
			$img              = wp_get_attachment_image_src( $site_icon, 'full' );
			$thumbnail        = wp_get_attachment_image_src( $site_icon, 'thumbnail' );

			$ciyashop_options['favicon_icon']['url']       = $img[0];
			$ciyashop_options['favicon_icon']['id']        = $site_icon;
			$ciyashop_options['favicon_icon']['width']     = $img[1];
			$ciyashop_options['favicon_icon']['height']    = $img[2];
			$ciyashop_options['favicon_icon']['thumbnail'] = $thumbnail[0];
			update_option( $opt_name, $ciyashop_options );
		}
	}

	return $response;
}

add_filter( 'redux/options/ciyashop_options/ajax_save/response', 'ciyashop_option_save' );
/**
 * Update customizer site icon from theme option favicon icon
 *
 * @param string $response .
 */
function ciyashop_option_save( $response ) {
	if ( isset( $response['options']['favicon_icon'] ) && ! empty( $response['options']['favicon_icon'] ) ) {
		$site_icon    = get_option( 'site_icon' );
		$favicon_icon = $response['options']['favicon_icon']['id'];
		if ( $favicon_icon !== $site_icon ) {
			update_option( 'site_icon', $favicon_icon );
		}
	}

	return $response;
}

/**
 * Get Product categories
 */
function ciyashop_get_product_categories() {
	$cat_titles = array();

	if ( is_admin() ) {
		$args = array(
			'type'         => 'post',
			'orderby'      => 'id',
			'order'        => 'DESC',
			'hide_empty'   => false,
			'hierarchical' => 1,
			'taxonomy'     => 'product_cat',
			'pad_counts'   => false,
		);

		$post_categories = get_categories( $args );

		foreach ( $post_categories as $cat ) {
			$cat_titles[ $cat->term_id ] = $cat->name;
		}
	}
	return $cat_titles;
}

if ( ! function_exists( 'ciyashop_array_unshift_assoc' ) ) {
	/**
	 * Array unshift
	 *
	 * @param array  $arr array.
	 * @param string $key key.
	 * @param string $val val.
	 */
	function ciyashop_array_unshift_assoc( &$arr, $key, $val ) {
		$arr         = array_reverse( $arr, true );
		$arr[ $key ] = $val;
		$arr         = array_reverse( $arr, true );
	}
}

/**
 * Allowed HTML
 *
 * @param string $allowed_els .
 */
function ciyashop_allowed_html( $allowed_els = '' ) {

	// bail early if parameter is empty.
	if ( empty( $allowed_els ) ) {
		return array();
	}

	if ( is_string( $allowed_els ) ) {
		$allowed_els = explode( ',', $allowed_els );
	}

	$allowed_html = array();

	$allowed_tags = wp_kses_allowed_html( 'post' );
	$svg_args     = array();
	foreach ( $allowed_els as $el ) {
		$el = trim( $el );
		if ( array_key_exists( $el, $allowed_tags ) ) {
			$allowed_html[ $el ] = $allowed_tags[ $el ];
		}
		if ( 'svg' === $el ) {
			$svg_args = array(
				'svg'   => array(
					'class'           => true,
					'aria-hidden'     => true,
					'aria-labelledby' => true,
					'role'            => true,
					'xmlns'           => true,
					'width'           => true,
					'height'          => true,
					'viewbox'         => true,
				),
				'g'     => array(
					'fill' => true,
				),
				'title' => array(
					'title' => true,
				),
				'path'  => array(
					'd'    => true,
					'fill' => true,
				),
			);
		}
	}
	if ( ! empty( $svg_args ) ) {
		$allowed_html = array_merge( $allowed_html, $svg_args );
	}
	return $allowed_html;
}

// Ciyashop auto complate search.
add_action( 'wp_ajax_ciyashop_auto_complete_search', 'ciyashop_auto_complete_search' );
add_action( 'wp_ajax_nopriv_ciyashop_auto_complete_search', 'ciyashop_auto_complete_search' );

if ( ! function_exists( 'ciyashop_auto_complete_search' ) ) {
	/**
	 * Auto compelete search
	 */
	function ciyashop_auto_complete_search() {
		global $ciyashop_options, $post;

		$ciyashop_nonce = isset( $_POST['ajax_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['ajax_nonce'] ) ) : '';

		if ( ! wp_verify_nonce( $ciyashop_nonce, 'ciyashop_nonce' ) ) {
			return false;
			exit();
		}

		$search_keyword  = isset( $_POST['search_keyword'] ) ? sanitize_text_field( wp_unslash( $_POST['search_keyword'] ) ) : '';
		$search_category = isset( $_POST['search_category'] ) ? sanitize_text_field( wp_unslash( $_POST['search_category'] ) ) : '';

		$post_titles = array();
		$data        = array();

		$taxonomy  = ( 'product' === $ciyashop_options['search_content_type'] ) ? 'product_cat' : 'category';
		$post_type = ( ! empty( $ciyashop_options['search_content_type'] ) ) ? $ciyashop_options['search_content_type'] : 'post';

		if ( 'all' === $post_type ) {
			$post_type = 'any';
		}

		$args = array(
			'post_type'      => $post_type,
			'post_status'    => 'publish',
			'posts_per_page' => -1, // phpcs:ignore WPThemeReview.CoreFunctionality.PostsPerPage.posts_per_page_posts_per_page
		);

		if ( ! empty( $search_keyword ) ) {
			$args['s'] = $search_keyword;
		}

		if ( ! empty( $search_category ) ) {
			$args['tax_query'][] = array(
				'taxonomy' => $taxonomy,
				'field'    => 'id',
				'terms'    => $search_category,
			);
		}

		if ( 'yes' === get_option( 'woocommerce_hide_out_of_stock_items' ) ) {
			$args['meta_query'][] = array(
				'key'     => '_stock_status',
				'value'   => 'outofstock',
				'compare' => 'NOT IN'
			);
		}

		/**
		 * Filters the autocomplete search arguments.
		 *
		 * @since 3.4.4
		 *
		 * @param     array       $args         An array of search arguments.
		 *
		 * @visible true
		 */
		$args = apply_filters( 'ciyashop_auto_complete_search_args', $args );

		$posts = new WP_Query( $args );

		if ( $posts->have_posts() ) {

			while ( $posts->have_posts() ) :
				$posts->the_post();
				$img    = get_the_post_thumbnail( $posts->ID, 'ciyashop-thumbnail-small' );
				$data[] = array(
					'post_title' => $post->post_title,
					'post_img'   => $img,
					'post_link'  => get_the_permalink( $post->ID ),
				);
			endwhile;

			wp_reset_postdata();
		}

		/**
		 * Filters the autocomplete search result.
		 *
		 * @since 3.4.4
		 *
		 * @param     array       $args         An array of search data.
		 * @param     array       $args         An array of search arguments.
		 *
		 * @visible true
		 */
		$data = apply_filters( 'ciyashop_auto_complete_search_data', $data, $args );

		echo wp_json_encode( $data );
		exit();
	}
}

add_filter( 'woocommerce_widget_cart_is_hidden', 'ciyashop_widget_cart_is_hidden' );
/**
 * Widget cart is hidden
 */
function ciyashop_widget_cart_is_hidden() {
	return false;
}
/**
 * @param string $plugin Plugin file.
 */
function ciyashop_check_plugin_active( $plugin = '' ) {

	if ( empty( $plugin ) ) {
		return false;
	}

	return ( in_array( $plugin, (array) get_option( 'active_plugins', array() ), true ) || ( function_exists( 'is_plugin_active_for_network' ) && is_plugin_active_for_network( $plugin ) ) );
}

add_action( 'ciyashop_before_footer', 'ciyashop_sticky_footer_mobile_device' );
/**
 * Sticky footer mobile device
 */
function ciyashop_sticky_footer_mobile_device() {
	global $ciyashop_options;

	if ( ( isset( $ciyashop_options['woocommerce_mobile_sticky_footer'] ) && $ciyashop_options['woocommerce_mobile_sticky_footer'] ) && wp_is_mobile() ) {
		get_template_part( 'template-parts/footer/sticky-footer-mobile' );
	}
}
/**
 * Is activated
 */
function ciyashop_is_activated() {
	$purchase_token = get_option( 'ciyashop_pgs_token' );
	if ( $purchase_token && ! empty( $purchase_token ) ) {
		return $purchase_token;
	}
	return false;
}

/**
 * Display admin notice if PGS Core is not updated
 */
function ciyashop_pgscore_update_notice() {

	$current_theme = wp_get_theme();

	if ( ! defined( 'PGSCORE_PATH' ) ) {
		return;
	}

	if (
		( ! defined( 'PGSCORE_VERSION' ) || ( defined( 'PGSCORE_VERSION' ) && version_compare( PGSCORE_VERSION, '4.7.0', '<' ) ) )
	) {
		$ciyashop_token = ciyashop_is_activated();
		if ( $ciyashop_token ) {
			$title     = esc_html__( 'Update PGS Core', 'ciyashop' );
			$msg       = '<p>' . __( 'Please ensure to update the PGS Core plugin to the latest version to enable features. If you have done customization, please take a complete backup of the website before updating the PGS Core plugin. In the latest version of the plugin, the dependency of Advanced Custom Fields Pro has been removed. So, the theme will work without using the ACF Pro plugin.', 'ciyashop' );
			$btn_title = esc_html__( 'Update PGS Core', 'ciyashop' );
			$url       = add_query_arg(
				array(
					'page'          => 'theme-plugins',
					'plugin_status' => 'update',
				),
				admin_url( 'themes.php' )
			);
		} else {
			$title     = esc_html__( 'Activate Theme', 'ciyashop' );
			$msg       = esc_html__( 'Please activate theme to enable the theme features.', 'ciyashop' );
			$btn_title = esc_html__( 'Activate Theme', 'ciyashop' );
			$url       = add_query_arg(
				array(
					'page' => 'ciyashop-panel',
				),
				admin_url( 'admin.php' )
			);

		}
		?>
		<div id="setting-error-pgscore_update" class="notice notice-error">
			<h2 class="setting-error-pgscore_update-header"><?php echo esc_html( $title ); ?></h2>
			<div class="setting-error-pgscore_update-desc"><?php echo wp_kses( $msg, array( 'p' => array() ) ); ?></div>
			<div class="setting-error-pgscore_update-action"><a href="<?php echo esc_url( $url ); ?>" class="button-primary"><?php echo esc_html( $btn_title ); ?></a></div>
		</div>
		<?php
	}
}
add_action( 'admin_notices', 'ciyashop_pgscore_update_notice' );

add_filter( 'ciyashop_content_container_classes', 'ciyashop_main_content_container_classes' );
/**
 * Portfolio container class
 *
 * @param mixed $classes  Classes.
 */
function ciyashop_main_content_container_classes( $classes ) {
	global $ciyashop_options;

	if ( is_post_type_archive( 'portfolio' ) && 'full_width' === $ciyashop_options['portfolio_sidebar'] && 1 === (int) $ciyashop_options['portfolio_fullscreen'] ) {
		$classes = array( 'container-fluid' );
	}
	return $classes;
}

/**
 * Related Portfolio
 */
function ciyashop_related_portfolio() {
	global $ciyashop_options;

	if ( isset( $ciyashop_options['display_related_portfolio'] ) && 1 === (int) $ciyashop_options['display_related_portfolio'] ) {
		get_template_part( 'template-parts/portfolio/single-related' );
	}
}

/**
 * Register Metabox
 */
function ciyashop_custome_sidebar_metabox() {
	add_meta_box(
		'pgs_custome_sidebar',
		__( 'Custome Sidebar', 'ciyashop' ),
		'pgs_custome_sidebar_callback',
		'page',
		'side'
	);
}

add_action( 'add_meta_boxes', 'ciyashop_custome_sidebar_metabox' ); // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped

/**
 * Metabox Dropdown
 *
 * @param WP_Post $post Post object.
 */
function pgs_custome_sidebar_callback( $post ) {
	wp_nonce_field( 'pgs_custom_sidebar_metabox_nonce', 'pgs_custom_sidebar_nonce' );
	$value = get_post_meta( $post->ID, 'pgs_custom_sidebar', true );
	?>
	<select name="pgs_custom_sidebar" id="pgs_custom_sidebar">
		<option value="">
			<?php echo esc_html( 'Default' ); ?>
		</option>
		<?php foreach ( $GLOBALS['wp_registered_sidebars'] as $sidebar ) { ?>
		<option value="<?php echo esc_attr( $sidebar['id'] ); ?>" <?php selected( $value, $sidebar['id'] ); ?>>
			<?php echo esc_html( ucwords( $sidebar['name'] ) ); ?>
		</option>
			<?php
		}
		?>
	</select>
	<?php
}

if ( ! function_exists( 'ciyashop_save_custome_sidebar_metabox' ) ) {
	/**
	 * Metabox Save
	 *
	 * @param int $post_id   Post ID.
	 */
	function ciyashop_save_custome_sidebar_metabox( $post_id ) {
		if ( ! isset( $_POST['pgs_custom_sidebar_nonce'] ) || ! wp_verify_nonce( sanitize_key( wp_unslash( $_POST['pgs_custom_sidebar_nonce'] ) ), 'pgs_custom_sidebar_metabox_nonce' ) ) {
			return;
		}

		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		if ( isset( $_POST['pgs_custom_sidebar'] ) ) {
			update_post_meta( $post_id, 'pgs_custom_sidebar', sanitize_text_field( wp_unslash( $_POST['pgs_custom_sidebar'] ) ) );
		}
	}
}
add_action( 'save_post', 'ciyashop_save_custome_sidebar_metabox' );

if ( ! function_exists( 'redux_search_options' ) ) {
	/**
	 * Redux Search Option
	 */
	function redux_search_options() {

		global $ciyashop_globals;

		if ( class_exists( 'Redux' ) ) {

			$option_fields = array();

			// Sections.
			$ciyashop_redux_sections = Redux::$sections;

			if ( empty( $ciyashop_redux_sections ) ) {
				return;
			}

			$redux_data = get_plugin_data( WP_PLUGIN_DIR . '/redux-framework/redux-framework.php' );

			$ciyashop_redux_sections = $ciyashop_redux_sections[ $ciyashop_globals['options_name'] ];

			if ( version_compare( $redux_data['Version'], '4.0', '<=' ) ) {
				$sections_count = ( Redux_Helpers::isWpDebug() ) ? count( $ciyashop_redux_sections ) + 2 : count( $ciyashop_redux_sections ) + 1;
			} else {
				$sections_count = ( Redux_Helpers::is_wp_debug() ) ? count( $ciyashop_redux_sections ) + 2 : count( $ciyashop_redux_sections ) + 1;
			}

			// Fields.
			$ciyashop_redux_fields = Redux::$fields;
			$ciyashop_redux_fields = $ciyashop_redux_fields[ $ciyashop_globals['options_name'] ];

			if ( version_compare( $redux_data['Version'], '4.0', '<=' ) ) {
				$fields_count = ( Redux_Helpers::isWpDebug() ) ? count( $ciyashop_redux_fields ) + 2 : count( $ciyashop_redux_fields ) + 1;
			} else {
				$fields_count = ( Redux_Helpers::is_wp_debug() ) ? count( $ciyashop_redux_fields ) + 2 : count( $ciyashop_redux_fields ) + 1;
			}

			$ciyashop_redux_sections['import/export'] = array(
				'title'    => 'Import / Export',
				'desc'     => '',
				'id'       => 'import/export',
				'icon'     => 'el el-refresh',
				'priority' => $sections_count,
			);

			$ciyashop_redux_fields['redux_import_export'] = array(
				'id'         => 'redux_import_export',
				'type'       => 'import_export',
				'full_width' => '1',
				'section_id' => 'import/export',
				'priority'   => $fields_count,
			);

			$ciyashop_redux_fields_new = array();

			if ( ! empty( $ciyashop_redux_fields ) && is_array( $ciyashop_redux_fields ) ) {
				foreach ( $ciyashop_redux_fields as $ciyashop_redux_field_k => $ciyashop_redux_field_data ) {
					$ciyashop_redux_fields_new[ $ciyashop_redux_field_data['section_id'] ][] = $ciyashop_redux_field_data;
				}
			}

			if ( ! empty( $ciyashop_redux_sections ) && is_array( $ciyashop_redux_sections ) ) {

				$section_id = 0;
				$index      = 0;

				$current_icon = '';

				foreach ( $ciyashop_redux_sections as $section ) {

					if ( isset( $section['title'] ) ) {
						$option_fields[ $index ]['title']      = $section['title'];
						$option_fields[ $index ]['id']         = $section['id'];
						$option_fields[ $index ]['path']       = $section['title'];
						$option_fields[ $index ]['section_id'] = $section['priority'];

						if ( isset( $section['icon'] ) ) {
							$current_icon = $option_fields[ $index ]['icon'] = $section['icon']; // phpcs:ignore Squiz.PHP.DisallowMultipleAssignments.Found
						} else {
							$option_fields[ $index ]['icon'] = $current_icon;
						}

						$index++;

						$section_fields = array();
						if ( isset( $ciyashop_redux_fields_new[ $section['id'] ] ) ) {
							$section_fields = $ciyashop_redux_fields_new[ $section['id'] ];
							foreach ( $section_fields as $field ) {
								$field_title = isset( $field['title'] ) ? trim( $field['title'] ) : false;
								if ( $field_title ) {
									$option_fields[ $index ]['title']      = $field['title'];
									$option_fields[ $index ]['id']         = $field['id'];
									$option_fields[ $index ]['path']       = $section['title'] . ' &raquo; ' . $field['title'];
									$option_fields[ $index ]['section_id'] = $section['priority'];

									if ( isset( $section['icon'] ) ) {
										$option_fields[ $index ]['icon'] = $section['icon'];
									} else {
										$option_fields[ $index ]['icon'] = $current_icon;
									}
									$index++;
								}
							}
						}
					}
				}
			}

			$localize_data['search_option_placeholder_text'] = esc_js( __( 'Search for Theme options', 'ciyashop' ) );
			$localize_data['reduxThemeOptions']              = $option_fields;

			/**
			 * Filters the list of theme option search items.
			 *
			 * @param     array       $localize_data      Theme option search localize data.
			 *
			 * @visible false
			 * @ignore
			 */
			return apply_filters( 'ciyashop_admin_search_options_localize_data', $localize_data );
		}
	}
}

if ( ! function_exists( 'ciyashop_iconpicker_icons' ) ) {
	/**
	 * Iconpicker icons
	 */
	function ciyashop_iconpicker_icons() {
		require_once get_parent_theme_file_path( '/includes/fontawesome-icons.php' );
		$fontawesome_icons = ciyashop_fontawesome_icons();
		return $fontawesome_icons;
	}
}

if ( ! function_exists( 'ciyashop_breadcrumbs' ) ) {
	/**
	 * Ciyashop breadcrumbs
	 *
	 * @param string $class .
	 */
	function ciyashop_breadcrumbs( $class = '' ) {
		global $ciyashop_options;

		// Do not display on the homepage.
		if ( is_front_page() ) {
			return;
		}

		// Get the query & post information.
		global $post, $wp_query;

		$breadcrums_id    = 'ciyashop_breadcrumbs';
		$breadcrums_class = 'ciyashop_breadcrumbs page-breadcrumb breadcrumbs';
		$home_title       = __( 'Home', 'ciyashop' );
		$parents          = '';

		if ( $class ) {
			$breadcrums_class .= ' ' . $class;
		}

		// If you have any custom post types with custom taxonomies, put the taxonomy name below (e.g. product_cat).
		$custom_taxonomy = array();

		$output  = '';
		$output .= '<ul id="' . $breadcrums_id . '" class="' . $breadcrums_class . '">';

		// Home page.
		$output .= '<li class="home"><span class="item-element"><a class="bread-link bread-home" href="' . get_home_url() . '" title="' . $home_title . '">' . $home_title . '</a></span></li>';

		if ( is_home() ) {
			$page_id = get_queried_object_id();
			$output .= '<li class="current-item item-home-page"><span class="item-element">';
			$output .= get_the_title( $page_id );
			$output .= '</span></li>';
		}
		if ( is_archive() && ! is_tax() && ! is_category() && ! is_tag() && ! is_author() ) {
			$output .= '<li class="current-item item-archive">';
			if ( is_day() ) :
				$output .= __( 'Daily Archives:', 'ciyashop' ) . ' <span class="item-element">' . get_the_date() . '</span>';
			elseif ( is_month() ) :
				$output .= __( 'Monthly Archives:', 'ciyashop' ) . ' <span class="item-element">' . get_the_date( _x( 'F Y', 'monthly archives date format', 'ciyashop' ) ) . '</span>';
			elseif ( is_year() ) :
				$output .= __( 'Yearly Archives:', 'ciyashop' ) . ' <span class="item-element">' . get_the_date( _x( 'Y', 'yearly archives date format', 'ciyashop' ) ) . '</span>';
			else :
				$output .= post_type_archive_title( $prefix = '', false ); // phpcs:ignore Squiz.PHP.DisallowMultipleAssignments.Found
			endif;
			$output .= '</li>';
		} elseif ( is_archive() && is_tax() && ! is_category() && ! is_tag() ) {
			// If post is a custom post type.
			$post_type = get_post_type();

			// If it is a custom post type display name and link.
			if ( 'post' !== $post_type ) {
				$post_type_object  = get_post_type_object( $post_type );
				$post_type_archive = get_post_type_archive_link( $post_type );
				$labels            = isset( $post_type_object->labels->name ) ? $post_type_object->labels->name : __( 'Archive Page', 'ciyashop' );

				$output .= '<li class="item-cat item-custom-post-type-' . $post_type . '"><span class="item-element"><a class="bread-cat bread-custom-post-type-' . $post_type . '" href="' . $post_type_archive . '" title="' . $labels . '">' . $labels . '</a></span></li>';
			}

			$custom_tax_name = get_queried_object()->name;
			$output         .= '<li class="current-item item-archive"><span class="item-element">' . $custom_tax_name . '</span></li>';
		} elseif ( is_single() ) {
			// If post is a custom post type.
			$post_type = get_post_type();

			// If it is a custom post type display name and link.
			if ( 'post' !== $post_type ) {
				$post_type_object  = get_post_type_object( $post_type );
				$post_type_archive = get_post_type_archive_link( $post_type );
				$output           .= '<li class="item-cat item-custom-post-type-' . $post_type . '"><span class="item-element"><a class="bread-cat bread-custom-post-type-' . $post_type . '" href="' . $post_type_archive . '" title="' . $post_type_object->labels->name . '">' . $post_type_object->labels->name . '</a></span></li>';
			}

			// Get post category info.
			$category = get_the_category();
			if ( 'product' === $post_type ) {
				$category = get_the_terms( get_the_ID(), 'product_cat' );
			}
			if ( ! empty( $category ) ) {
				// Get last category post is in.
				$values_of_category = array_values( $category );
				$last_category      = end( $values_of_category );
				// Get parent any categories and create array.
				$get_cat_parents = rtrim( get_category_parents( $last_category->term_id, true, ',' ), ',' );
				$cat_parents     = explode( ',', $get_cat_parents );
				$cat_display     = '';
				if ( 'product' === $post_type ) {
					$cats_id = array_column( get_the_terms( get_the_ID(), 'product_cat' ), 'term_id' );
					// Show only onw category.
					$cats_id = reset( $cats_id );
					if ( ! empty( $cats_id ) ) {
						$get_cat_parents = ( ! empty( $cats_id ) ) ? get_ancestors( $cats_id, 'product_cat' ) : '';
						$cat_link        = get_term_link( $cats_id );
						$last_cat        = get_term_by( 'id', $cats_id, 'product_cat' );
						if ( ! empty( $get_cat_parents ) ) {
							// Show only one category.
							$get_cat_parents  = get_term_by( 'id', $get_cat_parents[0], 'product_cat' );
							$parents_cat_link = get_term_link( $get_cat_parents->term_id, 'product_cat' );
							$cat_display     .= '<li class="item-cat"><span class="item-element"><a href="' . $parents_cat_link . '">' . $get_cat_parents->name . '</a></span></li>';
						}
						$cat_display .= '<li class="item-cat"><span class="item-element"><a href="' . $cat_link . '">' . $last_cat->name . '</a></span></li>';
					}
				}

				// Loop through parent categories and store in variable $cat_display.
				if ( ! empty( $cat_parents ) && is_array( $cat_parents ) ) {
					foreach ( $cat_parents as $parents ) {
						$cat_display .= ( empty( $parents ) ) ? '' : '<li class="item-cat"><span class="item-element">' . $parents . '</span></li>';
					}
				}
			}

			// If it's a custom post type within a custom taxonomy.
			foreach ( $custom_taxonomy as $taxonomy ) {
				$taxonomy_exists = taxonomy_exists( $taxonomy );
				if ( empty( $last_category ) && ! empty( $taxonomy ) && $taxonomy_exists ) {
					$taxonomy_terms = get_the_terms( $post->ID, $taxonomy );
					if ( isset( $taxonomy_terms[0] ) ) {
						$cat_id       = $taxonomy_terms[0]->term_id;
						$cat_nicename = $taxonomy_terms[0]->slug;
						$cat_link     = get_term_link( $taxonomy_terms[0]->term_id, $taxonomy );
						$cat_name     = $taxonomy_terms[0]->name;
					}
				}
			}

			// Check if the post is in a category.
			if ( ! empty( $last_category ) ) {
				$output .= $cat_display;
				$output .= '<li class="current-item item-' . $post->ID . '"><span class="item-element">' . get_the_title() . '</span></li>';
			} elseif ( ! empty( $cat_id ) ) {
				// Else if post is in a custom taxonomy.
				$output .= '<li class="item-cat item-cat-' . $cat_id . ' item-cat-' . $cat_nicename . '"><span class="item-element"><a class="bread-cat bread-cat-' . $cat_id . ' bread-cat-' . $cat_nicename . '" href="' . $cat_link . '" title="' . $cat_name . '">' . $cat_name . '</a></span></li>';
				$output .= '<li class="current-item item-' . $post->ID . '"><span class="item-element">' . get_the_title() . '</span></li>';
			} else {
				$output .= '<li class="current-item item-' . $post->ID . '"><span class="item-element">' . get_the_title() . '</span></li>';
			}
		} elseif ( is_category() ) {
			// Category page.
			$bread_cat_id   = get_cat_id( single_cat_title( '', false ) );
			$bread_cat      = get_category_parents( $bread_cat_id, 'true', ';' );
			$bread_cat      = explode( ';', $bread_cat );
			$bread_cat_size = sizeof( $bread_cat );
			$i              = 1;

			foreach ( $bread_cat as $bread ) {
				if ( $bread ) {
					if ( (int) $bread_cat_size === $i + 1 ) {
						$output .= '<li class="current-item item-cat"><span class="item-element">' . $bread . '</span></li>';
					} else {
						$output .= '<li class="current-item item-cat"><span class="item-element">' . $bread . '</span></li>';
					}
					$i++;
				}
			}
		} elseif ( is_page() ) {
			// Standard page.
			if ( $post->post_parent ) {
				// If child page, get parents.
				$anc = get_post_ancestors( $post->ID );

				// Get parents in the right order.
				$anc = array_reverse( $anc );

				// Parent page loop.
				foreach ( $anc as $ancestor ) {
					$parents .= '<li class="item-parent item-parent-' . $ancestor . '"><span class="item-element"><a class="bread-parent bread-parent-' . $ancestor . '" href="' . get_permalink( $ancestor ) . '" title="' . get_the_title( $ancestor ) . '">' . get_the_title( $ancestor ) . '</a></span></li>';
				}

				// Display parent pages.
				$output .= $parents;

				// Current page.
				$output .= '<li class="current-item item-' . $post->ID . '"><span class="item-element">' . get_the_title() . '</span></li>';
			} else {
				// Just display current page if not parents.
				$output .= '<li class="current-item item-' . $post->ID . '"><span class="item-element">' . get_the_title() . '</span></li>';
			}
		} elseif ( is_tag() ) {
			// Tag page.

			// Get tag information.
			$term_id  = get_query_var( 'tag_id' );
			$taxonomy = 'post_tag';
			$args     = 'include=' . $term_id;
			$terms    = get_terms( $taxonomy, $args );
			if ( array_filter( $terms ) ) {
				$get_term_id   = $terms[0]->term_id;
				$get_term_slug = $terms[0]->slug;
				$get_term_name = $terms[0]->name;

				// Display the tag name.
				$output .= '<li class="current-item item-tag-' . $get_term_id . ' item-tag-' . $get_term_slug . '"><span class="item-element">' . $get_term_name . '</span></li>';
			}
		} elseif ( is_day() ) {
			// Day archive.

			// Year link.
			$output .= '<li class="item-year item-year-' . get_the_time( 'Y' ) . '"><span class="item-element"><a class="bread-year bread-year-' . get_the_time( 'Y' ) . '" href="' . get_year_link( get_the_time( 'Y' ) ) . '" title="' . get_the_time( 'Y' ) . '">' . get_the_time( 'Y' ) . '&nbsp;' . __( 'Archives', 'ciyashop' ) . '</a></span></li>';

			// Month link.
			$output .= '<li class="item-month item-month-' . get_the_time( 'm' ) . '"><span class="item-element"><a class="bread-month bread-month-' . get_the_time( 'm' ) . '" href="' . get_month_link( get_the_time( 'Y' ), get_the_time( 'm' ) ) . '" title="' . get_the_time( 'M' ) . '">' . get_the_time( 'M' ) . '&nbsp;' . __( '.Archives', 'ciyashop' ) . '</a></span></li>';

			// Day display.
			$output .= '<li class="current-item item-' . get_the_time( 'j' ) . '"><span class="item-element">' . get_the_time( 'jS' ) . ' ' . get_the_time( 'M' ) . '&nbsp;' . __( 'Archives', 'ciyashop' ) . '</span></li>';
		} elseif ( is_month() ) {
			// Month Archive.

			// Year link.
			$output .= '<li class="item-year item-year-' . get_the_time( 'Y' ) . '"><span class="item-element"><a class="bread-year bread-year-' . get_the_time( 'Y' ) . '" href="' . get_year_link( get_the_time( 'Y' ) ) . '" title="' . get_the_time( 'Y' ) . '">' . get_the_time( 'Y' ) . '&nbsp;' . __( 'Archives', 'ciyashop' ) . '</a></span></li>';

			// Month display.
			$output .= '<li class="item-month item-month-' . get_the_time( 'm' ) . '"><span class="item-element">' . get_the_time( 'M' ) . '&nbsp;' . __( 'Archives', 'ciyashop' ) . '</span></li>';
		} elseif ( is_year() ) {
			// Display year archive.
			$output .= '<li class="current-item item-current-' . get_the_time( 'Y' ) . '"><span class="item-element">' . get_the_time( 'Y' ) . '&nbsp;' . __( 'Archives', 'ciyashop' ) . '</span></li>';
		} elseif ( is_author() ) {
			// Auhor archive.
			// Get the author information.
			global $author;
			$userdata = get_userdata( $author );
			// Display author name.
			$output .= '<li class="current-item item-current-' . $userdata->user_nicename . '"><span class="item-element">' . ' ' . $userdata->display_name . '</span></li>';
		} elseif ( get_query_var( 'paged' ) && ! is_home() ) {
			// Paginated archives.
			$output .= '<li class="current-item item-current-' . get_query_var( 'paged' ) . '"><span class="item-element">' . __( 'Page', 'ciyashop' ) . ' ' . get_query_var( 'paged' ) . '</span></li>';
		} elseif ( is_search() ) {
			// Search results page.
			$output .= '<li class="current-item item-current-' . get_search_query() . '"><span class="item-element">' . __( 'Search results for', 'ciyashop' ) . ': ' . get_search_query() . '</span></li>';
		}

		$output .= '</ul>';

		return $output;
	}
}

if ( ! function_exists( 'ciyashop_create_google_fonts_url' ) ) {
	/**
	 * Create Google Fonts url
	 */
	function ciyashop_create_google_fonts_url( $fonts ) {

		$link     = '';
		$subsets  = array();
		$protocol = ( is_ssl() ) ? 'https:' : 'http:';

		foreach ( $fonts as  $font ) {
			if ( ! empty( $font['font-family'] ) ) {

				if ( ! empty( $link ) ) {
					$link .= '%7C'; // Append a new font to the string.
				}

				$link .= $font['font-family'];

				if ( ! empty( $font['font-style'] ) || ! empty( $font['font-weight'] ) || ! empty( $font['all-styles'] ) ) {
					$link .= ':';

					if ( ! empty( $font['all-styles'] ) ) {
						$link .= implode( ',', $font['all-styles'] );
					} elseif ( ! empty( $font['font-weight'] ) || ! empty( $font['font-style'] ) ) {

						if ( ! empty( $font['font-weight'] ) ) {
							$link .= $font['font-weight'];
						}

						if ( ! empty( $font['font-weight'] ) ) {
							$link .= $font['font-style'];
						}
					}
				}

				if ( ! empty( $font['subsets'] ) ) {
					if ( is_array( $font['subsets'] ) ) {
						foreach ( $font['subsets'] as $subset ) {
							if ( ! in_array( $subset, $subsets, true ) ) {
								array_push( $subsets, $subset );
							}
						}
					} else {
						array_push( $subsets, $font['subsets'] );
					}
				}
			}
		}

		if ( ! empty( $subsets ) ) {
			$link .= '&subset=' . implode( ',', $subsets );
		}

		if ( empty( $link ) ) {
			return;
		}

		return $protocol . '//fonts.googleapis.com/css?family=' . str_replace( '|', '%7C', $link );
	}
}

if ( ! function_exists( 'ciyashop_get_tooltip_position' ) ) {
	/**
	 * Get tooltip position
	 */
	function ciyashop_get_tooltip_position() {
		global $ciyashop_options;

		$product_hover_style = ciyashop_product_hover_style();
		$positions           = array();

		switch ( $product_hover_style ) {
			case 'default':
				$cart_position       = 0;
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 'right';
				break;
			case 'icon-top-left':
				$cart_position       = 'right';
				$wishlist_position   = 'left';
				$compare_position    = 'right';
				$quick_view_position = 'right';
				break;
			case 'icons-top-right':
				$cart_position       = 'left';
				$wishlist_position   = 'left';
				$compare_position    = 'left';
				$quick_view_position = 'left';
				break;
			case 'image-icon-left':
				$cart_position       = 'right';
				$wishlist_position   = 'right';
				$compare_position    = 'right';
				$quick_view_position = 0;
				break;
			case 'image-icon-bottom':
				$cart_position       = 'top';
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 0;
				break;
			case 'image-center':
				$cart_position       = 'top';
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 0;
				break;
			case 'image-left':
				$cart_position       = 0;
				$wishlist_position   = 'right';
				$compare_position    = 'right';
				$quick_view_position = 'right';
				break;
			case 'button-standard':
				$cart_position       = 0;
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 'top';
				break;
			case 'icons-left':
				$cart_position       = 0;
				$wishlist_position   = 'right';
				$compare_position    = 'right';
				$quick_view_position = 'right';
				break;
			case 'icons-rounded':
				$cart_position       = 0;
				$wishlist_position   = 'right';
				$compare_position    = 'right';
				$quick_view_position = 'right';
				break;
			case 'icons-bottom-right':
				$cart_position       = 'left';
				$wishlist_position   = 'left';
				$compare_position    = 'left';
				$quick_view_position = 'left';
				break;
			case 'image-bottom':
				$cart_position       = 'top';
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 'top';
				break;
			case 'image-bottom-2':
				$cart_position       = 'top';
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 'top';
				break;
			case 'image-bottom-bar':
				$cart_position       = 0;
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 0;
				break;
			case 'info-bottom':
				$cart_position       = 0;
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 0;
				break;
			case 'info-bottom-bar':
				$cart_position       = 0;
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 0;
				break;
			case 'hover-summary':
				$cart_position       = 0;
				$wishlist_position   = 'right';
				$compare_position    = 'top';
				$quick_view_position = 'top';
				break;
			case 'minimal-hover-cart':
				$cart_position       = 0;
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 'top';
				break;
			case 'minimal':
				$cart_position       = 0;
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 0;
				break;
			case 'info-transparent-center':
				$cart_position       = 0;
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 'top';
				break;
			case 'icons-transparent-center':
				$cart_position       = 0;
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 'top';
				break;
			case 'standard-info-transparent':
				$cart_position       = 'left';
				$wishlist_position   = 'left';
				$compare_position    = 'left';
				$quick_view_position = 'left';
				break;
			case 'standard-quick-shop':
				$cart_position       = 0;
				$wishlist_position   = 'left';
				$compare_position    = 'left';
				$quick_view_position = 'left';
				break;
			case 'grocery-1':
				$cart_position       = 0;
				$wishlist_position   = 'right';
				$compare_position    = 'right';
				$quick_view_position = 'right';
				break;
			case 'grocery-2':
				$cart_position       = 0;
				$wishlist_position   = 'right';
				$compare_position    = 'right';
				$quick_view_position = 'right';
				break;
			default:
				$cart_position       = 'top';
				$wishlist_position   = 'top';
				$compare_position    = 'top';
				$quick_view_position = 'top';
		}

		$positions = array(
			'cart_icon'       => $cart_position,
			'wishlist_icon'   => $wishlist_position,
			'compare_icon'    => $compare_position,
			'quick_view_icon' => $quick_view_position,
		);

		return $positions;
	}
}

add_action( 'ciyashop_before_page_wrapper', 'ciyashop_before_page_wrapper_check' );

if ( ! function_exists( 'ciyashop_before_page_wrapper_check' ) ) {
	/**
	 * Before page wrapper check
	 */
	function ciyashop_before_page_wrapper_check() {
		$ciyashop_theme_options = get_option( 'ciyashop_options' );
		global $cs_product_list_styles, $ciyashop_options;

		if ( class_exists( 'WooCommerce' ) ) {
			$product_hover_style = ciyashop_product_hover_style();
			if ( $product_hover_style ) {
				switch ( $product_hover_style ) {
					case $cs_product_list_styles['default']:
						add_action( 'woocommerce_before_shop_loop_item_title', 'ciyashop_product_actions_add_quick_view_link', 20 );
						break;
					case $cs_product_list_styles['image-icon-left']:
						add_action( 'woocommerce_before_shop_loop_item_title', 'ciyashop_product_actions_add_quick_view_link', 20 );
						break;
					case $cs_product_list_styles['image-icon-bottom']:
						add_action( 'woocommerce_before_shop_loop_item_title', 'ciyashop_product_actions_add_quick_view_link', 20 );
						break;
					case $cs_product_list_styles['icons-center']:
						add_action( 'woocommerce_before_shop_loop_item_title', 'ciyashop_product_actions_add_quick_view_link', 20 );
						break;
					case $cs_product_list_styles['minimal']:
						add_action( 'woocommerce_before_shop_loop_item_title', 'ciyashop_product_actions_add_quick_view_link', 20 );
						break;
					case $cs_product_list_styles['icons-bottom-bar']:
						add_action( 'woocommerce_before_shop_loop_item_title', 'ciyashop_product_actions_add_quick_view_link', 20 );
						break;
					case $cs_product_list_styles['info-bottom']:
						add_action( 'woocommerce_before_shop_loop_item_title', 'ciyashop_product_actions_add_quick_view_link', 20 );
						break;
					case $cs_product_list_styles['info-bottom-bar']:
						add_action( 'woocommerce_before_shop_loop_item_title', 'ciyashop_product_actions_add_quick_view_link', 20 );
						break;
					default:
						add_action( 'ciyashop_product_actions', 'ciyashop_product_actions_add_quick_view_link', 20 );
				}
			}
		}
	}
}

if ( ! function_exists( 'ciyashop_product_actions_add_quick_view_link' ) ) {
	/**
	 * Product actions add quick view link
	 */
	function ciyashop_product_actions_add_quick_view_link() {
		global $post, $ciyashop_options;

		if ( ! $ciyashop_options['quick_view'] ) {
			return;
		}

		$position                 = ciyashop_get_tooltip_position();
		$quick_view_icon_position = isset( $position['quick_view_icon'] ) ? $position['quick_view_icon'] : '';
		?>
		<div class="product-action product-action-quick-view">
			<?php
			if ( $quick_view_icon_position ) {
				?>
				<a href="<?php echo esc_url( get_the_permalink( $post->ID ) ); ?>" class="open-quick-view" data-id="<?php echo esc_attr( $post->ID ); ?>" data-toggle='tooltip' data-original-title="<?php esc_attr_e( 'Quick view', 'ciyashop' ); ?>" data-placement='<?php echo esc_attr( $quick_view_icon_position ); ?>'>
					<?php esc_html_e( 'Quick View', 'ciyashop' ); ?>
				</a>
				<?php
			} else {
				?>
				<a href="<?php echo esc_url( get_the_permalink( $post->ID ) ); ?>" class="open-quick-view" data-id="<?php echo esc_attr( $post->ID ); ?>">
					<?php esc_html_e( 'Quick View', 'ciyashop' ); ?>
				</a>
				<?php
			}
			?>
		</div>
		<?php
	}
}

if ( ! function_exists( 'ciyashop_get_selected_menu_id' ) ) {
	/**
	 * Get selected menu id
	 */
	function ciyashop_get_selected_menu_id() {

		$nav_menus = wp_get_nav_menus( array( 'orderby' => 'name' ) );

		$menu_count = count( $nav_menus );

		$nav_menu_selected_id = isset( $_REQUEST['menu'] ) ? (int) $_REQUEST['menu'] : 0;

		$add_new_screen = ( isset( $_GET['menu'] ) && 0 == $_GET['menu'] ) ? true : false; // phpcs:ignore WordPress.PHP.StrictComparisons.LooseComparison

		// Get recently edited nav menu.
		$recently_edited = absint( get_user_option( 'nav_menu_recently_edited' ) );
		if ( empty( $recently_edited ) && is_nav_menu( $nav_menu_selected_id ) ) {
			$recently_edited = $nav_menu_selected_id;
		}

		// Use $recently_edited if none are selected.
		if ( empty( $nav_menu_selected_id ) && ! isset( $_GET['menu'] ) && is_nav_menu( $recently_edited ) ) {
			$nav_menu_selected_id = $recently_edited;
		}

		// On deletion of menu, if another menu exists, show it.
		if ( ! $add_new_screen && 0 < $menu_count && isset( $_GET['action'] ) && 'delete' === $_GET['action'] ) {
			$nav_menu_selected_id = $nav_menus[0]->term_id;
		}

		if ( empty( $nav_menu_selected_id ) && ! empty( $nav_menus ) && ! $add_new_screen ) {
			// if we have no selection yet, and we have menus, set to the first one in the list.
			$nav_menu_selected_id = $nav_menus[0]->term_id;
		}

		return $nav_menu_selected_id;

	}
}

add_action( 'wp_enqueue_scripts', 'ciyashop_404_wpbakery_css', 999 );
/**
 * Function to output shortcodes CSS and custom CSS on 404 page.
 *
 * @since 3.5.0
 */
function ciyashop_404_wpbakery_css() {
	global $ciyashop_options;

	if ( ! is_404() ) {
		return;
	}

	if ( ! isset( $ciyashop_options['fourofour_page_content_source'] ) || ( isset( $ciyashop_options['fourofour_page_content_source'] ) && 'page' !== $ciyashop_options['fourofour_page_content_source'] ) ) {
		return;
	}

	if ( ! isset( $ciyashop_options['fourofour_page_content_source'] ) || ( isset( $ciyashop_options['fourofour_page_content_page'] ) && '' === $ciyashop_options['fourofour_page_content_page'] ) ) {
		return;
	}

	$fourofour_content_page_id = $ciyashop_options['fourofour_page_content_page'];
	$fourofour_content_page    = get_post( $fourofour_content_page_id );

	if ( ! $fourofour_content_page || 'publish' !== $fourofour_content_page->post_status ) {
		return;
	}

	$wpbakery_css = '';

	if ( $fourofour_content_page_id ) {
		if ( ( function_exists( 'vc_get_param' ) && 'true' === vc_get_param( 'preview' ) ) && wp_revisions_enabled( get_post( $fourofour_content_page_id ) ) ) {
			$latest_revision = wp_get_post_revisions( $fourofour_content_page_id );
			if ( ! empty( $latest_revision ) ) {
				$array_values              = array_values( $latest_revision );
				$fourofour_content_page_id = $array_values[0]->ID;
			}
		}

		$shortcodes_custom_css = get_metadata( 'post', $fourofour_content_page_id, '_wpb_shortcodes_custom_css', true );
		if ( ! empty( $shortcodes_custom_css ) ) {
			$shortcodes_custom_css = wp_strip_all_tags( $shortcodes_custom_css );
			$wpbakery_css         .= $shortcodes_custom_css;
		}

		$post_custom_css = get_metadata( 'post', $fourofour_content_page_id, '_wpb_post_custom_css', true );
		if ( ! empty( $post_custom_css ) ) {
			$post_custom_css = wp_strip_all_tags( $post_custom_css );
			$wpbakery_css   .= $post_custom_css;
		}
	}
	if ( ! empty( $wpbakery_css ) ) {
		wp_add_inline_style( 'ciyashop-style', $wpbakery_css );
	}
}

/**
 * Check whether custom language flag exists.
 *
 * @param string $flag_url Language flag URL.
 *
 * @return bool
 */
function ciyashop_wpml_custom_flag_exists( $flag_url = '' ) {

	if ( empty( $flag_url ) ) {
		return $flag_url;
	}

	$uploads = wp_upload_dir( null, false );
	$basedir = $uploads['basedir'];
	$baseurl = $uploads['baseurl'];

	if ( false === strpos( $flag_url, $baseurl ) ) {
		return true;
	}

	$flag_path = str_replace( $baseurl, $basedir, $flag_url );

	return ( file_exists( $flag_path ) );
}

/**
 * Get post by title.
 */
function ciyashop_get_page_by_title( $title ) {
	$posts = get_posts(
		array(
			'post_type'              => 'page',
			'title'                  => $title,
			'post_status'            => 'all',
			'numberposts'            => 1,
			'update_post_term_cache' => false,
			'update_post_meta_cache' => false,
			'orderby'                => 'post_date ID',
			'order'                  => 'ASC',
		)
	);

	if ( ! empty( $posts ) ) {
		$page_got_by_title = $posts[0];
	} else {
		$page_got_by_title = null;
	}

	return $page_got_by_title;
}

function ciyashop_get_theme_color( $color = '' ) {
	global $ciyashop_options;

	$color_value = '';
	$colors      = apply_filters(
		'ciyashop_get_theme_color',
		array(
			'primary_color'   => '#04d39f',
			'secondary_color' => '#323232',
			'tertiary_color'  => '#777777',
		)
	);

	if ( ! empty( $color ) ) {
		$color       = ( isset( $color ) ) ? $color : 'primary_color';
		$color_value = ( isset( $ciyashop_options[ $color ] ) && ! empty( $ciyashop_options[ $color ] ) ) ? $ciyashop_options[ $color ] : $colors[ $color ];
	}

	return $color_value;
}

function ciyashop_body_styles() {
	$primary_color   = ciyashop_get_theme_color( 'primary_color' );
	$secondary_color = ciyashop_get_theme_color( 'secondary_color' );
	$tertiary_color  = ciyashop_get_theme_color( 'tertiary_color' );

	$css_styles      = array();
	$css_string      = '';

    $variable_defaults = array(
		'cs-primary-color'   => $primary_color,
		'cs-secondary-color' => $secondary_color,
		'cs-tertiary-color'  => $tertiary_color,
	);

	$css_vars       = apply_filters( 'ciyashop_body_css_variables', array() );
	$css_vars       = wp_parse_args( $css_vars, $variable_defaults );
	$css_var_parsed = ciyashop_css_var_parser( $css_vars );
	$css_styles     = array_merge( $css_styles, $css_var_parsed );

	$styles = apply_filters( 'ciyashop_body_styles', array() );

	if ( is_array( $styles ) && ! empty( $styles ) ) {
		foreach ( $styles as $property => $value ) {
			$css_styles[] = "$property: $value;";
		}
    }

	if ( is_array( $css_styles ) && ! empty( $css_styles ) ) {
		$css_string = implode( ';', $css_styles );
		echo 'style="' . esc_attr( $css_string ) . '"'; // Print the CSS string.
	}
}

/**
 * Parse CSS variables with provided array.
 *
 * @param array $css_vars An array of CSS variable names and values.
 * @return array The generated CSS variables with modified or prefixed names.
 */
function ciyashop_css_var_parser( $css_vars = array() ) {
	$result = array();
	$css_vars_parsed = array();

	// Check if the input is a non-empty array.
	if ( is_array( $css_vars ) && ! empty( $css_vars ) ) {

		// Modify or prefix CSS variable names.
		$parsed_css_vars = array_map( function( $key ) {
			if ( substr( $key, 0, 2) === '--' ) {
				return $key;
			} elseif ( substr( $key, 0, 1) === '-' ) {
				return '--' . substr( $key, 1 );
			} else {
				return '--' . $key;
			}
		}, array_keys( $css_vars ) );

		// Combine modified CSS variable names with original values.
		$css_vars_parsed = array_combine( $parsed_css_vars, array_values( $css_vars ) );
	}

	if ( ! empty( $css_vars_parsed ) ) {
		foreach ( $css_vars_parsed as $property => $value ) {
			$result[] = "$property: $value";
		}
	}

	return $result;
}

function ciyashop_mobile_sticky_footer_elements() {
	$elements = array(
		'home'       => array(
			'title'      => esc_html__( 'Home', 'ciyashop' ),
			'icon_class' => 'vc_icon_element-icon glyph-icon pgsicon-ecommerce-house',
		),
		'wishlist'   => array(
			'title'      => esc_html__( 'Wishlist', 'ciyashop' ),
			'icon_class' => 'vc_icon_element-icon glyph-icon pgsicon-ecommerce-heart',
		),
		'myaccount' => array(
			'title'      => esc_html__( 'My Account', 'ciyashop' ),
			'icon_class' => 'vc_icon_element-icon glyph-icon pgsicon-ecommerce-avatar',
		),
		'cart'       => array(
			'title'      => esc_html__( 'Cart', 'ciyashop' ),
			'icon_class' => 'vc_icon_element-icon glyph-icon pgsicon-ecommerce-commerce-3',
		),
	);
	$elements = apply_filters( 'ciyashop_mobile_sticky_footer_elements', $elements );
	return $elements;
}

function ciyashop_mobile_sticky_footer_elements_options() {
	$elements = ciyashop_mobile_sticky_footer_elements();
	$elements = array_map( function( $element ) {
		$return = $element['title'];
		return $return;
	}, $elements );
	return $elements;
}
